/*----------------------------------------------------------------------------------------*
 * SurfaceFrame.java                                                                      *
 *                                                                                        *
 * Surface Plotter   version 1.10    14 Oct 1996                                          *
 *                   version 1.20     8 Nov 1996                                          *
 *                   version 1.30b1  17 May 1997                                          *
 *                   bug fixed       21 May 1997                                          *
 *                   version 1.30b2  18 Oct 2001                                          *
 *                                                                                        *
 * Copyright (c) Yanto Suryono <yanto@fedu.uec.ac.jp>                                     *
 *                                                                                        *
 *                                                                                        *
 * MathOptAnimator   version 1.00     10 Feb 2002                                         *
 * Copyright (c) 2003 Ramazan Buzdemir <buzdemir@zpr.uni-koeln.de>                        *
 *                                                                                        *
 *                                                                                        *
 * This program is free software; you can redistribute it and/or modify it                *
 * under the terms of the GNU General Public License as published by the                  *
 * Free Software Foundation; either version 2 of the License, or (at your option)         *
 * any later version.                                                                     *
 *                                                                                        *
 * This program is distributed in the hope that it will be useful, but                    *
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or          *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for               *
 * more details.                                                                          *
 *                                                                                        *
 * You should have received a copy of the GNU General Public License along                *
 * with this program; if not, write to the Free Software Foundation, Inc.,                *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA                                  *
 *                                                                                        *
 *----------------------------------------------------------------------------------------*/

import java.io.*;
import java.awt.*;
import java.net.*;
import java.util.*;

/**
 * The class <code>SurfaceFrame</code> produces a frame for Surface Plotter applet.
 * It also provides Surface Plotter a pulldown menu and acts as a middle man between
 * <code>SurfaceCanvas</code> class and other Surface Plotter components
 *
 * @author  Yanto Suryono
 */
 
public final class SurfaceFrame extends Frame {

  public FunctionPanel function_panel;          // the function panel
  public SettingPanel setting_panel;            // the setting panel
  public AlgorithmPanel algorithm_panel;        // the algorithm panel  
  public SurfaceCanvas canvas;                  // the drawing canvas
  public SurfacePlotter applet;                 // the applet, if present
  public Parser parser;                         // the function parser 
  
  private String function;
  private SurfaceVertex[] vertex;
  private int calc_divisions;                   // number of divisions to calculate
  private MenuBar menubar;
  
  private static final int MODES_COUNT = 4;
  
  /**
   * The constructor of <code>SurfaceFrame</code>
   *
   * @param applet the Surface Plotter applet
   * @see   SurfacePlotter
   */
   
  SurfaceFrame(SurfacePlotter applet) {
    this.applet = applet;	

    function = "";

    parser = new Parser(2);
    parser.defineVariable(1,"x");
    parser.defineVariable(2,"y");
    
    setTitle(SurfacePlotter.APP_NAME + " " + SurfacePlotter.APP_VERSION);
    setFont(new Font("Helvetica", Font.PLAIN, 11));
    setBackground(Color.lightGray); 
    setLayout(new BorderLayout());
    
    add("North", function_panel = new FunctionPanel(this));
    add("East", new SurfaceBorder(setting_panel = new SettingPanel(this),5,5,false));  
    add("Center", new SurfaceBorder(canvas = new SurfaceCanvas(this),false));
    add("South", new SurfaceBorder(algorithm_panel = new AlgorithmPanel(this),5,5,false));

    setMenuBar(menubar = createMenuBar());
    pack(); show();

    if (algorithm_panel.getFunctionType() == "quadratisch") 
      algorithm_panel.setDiffVisible(false);
                                 
    canvas.repaint();       
    requestFocus();
  }              
                 
  /**
   * Parses defined functions and calculates surface vertices 
   */
    
  public boolean plot(boolean parse, boolean paint) {
    
    float   stepx, stepy, x, y, v;
    float   xi,xx,yi,yx,zi,zx;
    float   min, max;
    int     i,j,k,total;

    setting_panel.setEnabled(false);
    
//-----------------------------------------------------------------------------------------------
    try {
      xi = Float.valueOf(setting_panel.getXMin()).floatValue();
      xx = Float.valueOf(setting_panel.getXMax()).floatValue();      
      if (xi >= xx) throw new NumberFormatException();
    }
    catch(NumberFormatException e) {
      function_panel.setErrorMessage("Fehler: es muss x-min < x-max gelten");
      setting_panel.setEnabled(true);
      return false;
    }
    
    try {
      yi = Float.valueOf(setting_panel.getYMin()).floatValue();
      yx = Float.valueOf(setting_panel.getYMax()).floatValue();
      if (yi >= yx) throw new NumberFormatException();
    }
    catch(NumberFormatException e) {
      function_panel.setErrorMessage("Fehler: es muss y-min < y-max gelten");
      setting_panel.setEnabled(true);
      return false;
    }    
    
    try {
      zi = Float.valueOf(setting_panel.getZMin()).floatValue();
      zx = Float.valueOf(setting_panel.getZMax()).floatValue();
      if (zi >= zx) throw new NumberFormatException();
    }
    catch(NumberFormatException e) {
      function_panel.setErrorMessage("Fehler: es muss z-min < z-max gelten");
      setting_panel.setEnabled(true);
      return false;
    }    
//-----------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------
    if (parse) {
      function_panel.setMessage("lese Funktion ein ...");              
    
      parser.define(function_panel.getFunctionDefinition());
      parser.parse();
   
      if (parser.getErrorCode() != parser.NO_ERROR) {
        function_panel.setErrorMessage("Fehler beim Einlesen der Funktion: " + 
                                       parser.getErrorString() + 
                                       " an Position " + parser.getErrorPosition());
        function_panel.setErrorPosition(parser.getErrorPosition());
        setting_panel.setEnabled(true);
        return false;
      }
      
      if (!function.equals(function_panel.getFunctionDefinition())) {
        algorithm_panel.algorithm.points_x = null;
        algorithm_panel.algorithm.points_y = null;
        function = function_panel.getFunctionDefinition();
      }
    }
//-----------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------
    if (paint) {
    	        	
      calc_divisions = setting_panel.getCalcDivisions();    
      total = (calc_divisions+1)*(calc_divisions+1);

      allocateMemory(total);
      if (vertex == null) {
        setting_panel.setEnabled(true); 
        return false;
      }
    
      stepx = (xx - xi) / calc_divisions; 
      stepy = (yx - yi) / calc_divisions;    
    
      max = Float.NaN;
      min = Float.NaN;
        
      i = 0; j = 0; k = 0; x = xi; y = yi; 
    
      float xfactor = 20/(xx-xi);
      float yfactor = 20/(yx-yi);

      while (i <= calc_divisions) {
        parser.setVariable(1,x);
        parser.setVariable(2,y);
        while (j <= calc_divisions) {
          v = (float)parser.evaluate();       
          if (Float.isInfinite(v)) v = Float.NaN;
          if (!Float.isNaN(v)) {
            if (Float.isNaN(max) || (v > max)) max = v; else
            if (Float.isNaN(min) || (v < min)) min = v;
          }
          vertex[k] = new SurfaceVertex((x-xi)*xfactor-10,
                                        (y-yi)*yfactor-10,v);
          j++; y += stepy;
          parser.setVariable(2,y);         
          k++;    
          function_panel.setMessage("berechne : " + k*100/total + "% fertig");
        }
        j = 0; y = yi; i++; x += stepx;
      }
    
      canvas.destroyImage();
      canvas.setRanges(xi,xx,yi,yx); 
      canvas.setValuesArray(vertex);
      canvas.setDataAvailability(true);   
      canvas.repaint();
    }
//-----------------------------------------------------------------------------------------------

      setting_panel.setEnabled(true);
      return true;
  }

  /**
   * Handles events. 
   *
   * @param e the event
   */

  public boolean handleEvent(Event e) {	
    if ((e.id == Event.WINDOW_DESTROY) && (e.target == this)) {  
      if (applet == null) {
        dispose();
        System.exit(0);
      }
      else 
        applet.disposeWindow();
    }
    else
      if (e.target instanceof MenuItem) 
        processMenuEvent((MenuItem)e.target);
    else
      return super.handleEvent(e);
    return true;
  }

  /**
   * Determines whether to show bounding box.
   *
   * @return <code>true</code> if to show bounding box
   */
   
  public boolean isBoxed() {
    return ((CheckboxMenuItem)getMenuItem(OPT_BOXED)).getState();
  }

  /**
   * Determines whether to show x-y mesh.
   *
   * @return <code>true</code> if to show x-y mesh
   */

  public boolean isMesh() {
    return ((CheckboxMenuItem)getMenuItem(OPT_MESH)).getState();
  }

  /**
   * Determines whether to scale axes and bounding box.
   *
   * @return <code>true</code> if to scale bounding box
   */

  public boolean isScaleBox() {
    return ((CheckboxMenuItem)getMenuItem(OPT_SCALE)).getState();
  }

  /**
   * Determines whether to show x-y ticks.
   *
   * @return <code>true</code> if to show x-y ticks
   */

  public boolean isDisplayXY() {
    return ((CheckboxMenuItem)getMenuItem(OPT_XYTICKS)).getState();
  }

  /**
   * Determines whether to show z ticks.
   *
   * @return <code>true</code> if to show z ticks
   */

  public boolean isDisplayZ() {
    return ((CheckboxMenuItem)getMenuItem(OPT_ZTICKS)).getState();
  }
  
  /**
   * Determines whether to show face grids.
   *
   * @return <code>true</code> if to show face grids
   */

  public boolean isDisplayGrids() {
    return ((CheckboxMenuItem)getMenuItem(OPT_FACEGRIDS)).getState();
  }
  
  /**
   * Gets the selected shading/plot mode
   *
   * @return the selected shading/plot mode
   */
   
  public int getPlotMode() {    
    for (int i=0; i < MODES_COUNT; i++) {   
      if (((CheckboxMenuItem)getMenuItem(OPT_SHADES+i)).getState())
      return i;
    }
    return 0;
  }

  /**
   * Creates menu bar
   *
   * @return the menu bar
   */
   
  private MenuBar createMenuBar() {
    MenuBar mb = new MenuBar();
    Menu menu;
     
    mb.add(menu = new Menu("Datei"));
    menu.add("Beenden");

    mb.add(menu = new Menu("Optionen"));
    menu.add(new CheckboxMenuItem("Zeige Begrenzungsbox"));          // 0
    menu.add(new CheckboxMenuItem("Zeige X-Y Maschen"));             // 1
    menu.add(new CheckboxMenuItem("Zeige Gitternetzlinien"));        // 2
    menu.add(new CheckboxMenuItem("Skaliere Begrenzungsbox"));       // 3
    menu.addSeparator();                                             // 4
    menu.add(new CheckboxMenuItem("Zeige X-Y-Achsenbeschriftung"));  // 5
    menu.add(new CheckboxMenuItem("Zeige Z-Achsenbeschriftung"));    // 6
    menu.addSeparator();                                             // 7
    menu.add(new CheckboxMenuItem("Einfarbige Darstellung"));         // 8
    menu.add(new CheckboxMenuItem("Farbspektrum Darstellung"));      // 9
    menu.add(new CheckboxMenuItem("Graustufen Darstellung"));        // 10
    menu.add(new CheckboxMenuItem("Schattierte Darstellung"));        // 11
            
    // Initial states
    
    ((CheckboxMenuItem)menu.getItem(0)).setState(true);
    ((CheckboxMenuItem)menu.getItem(1)).setState(true);  
    ((CheckboxMenuItem)menu.getItem(3)).setState(true);
    ((CheckboxMenuItem)menu.getItem(5)).setState(true);  
    ((CheckboxMenuItem)menu.getItem(6)).setState(true);          
    ((CheckboxMenuItem)menu.getItem(10)).setState(true);

    Menu helpmenu = new Menu("Hilfe");
    //helpmenu.add("Men Kommandos");
    helpmenu.add("Funktions Syntax");
    //helpmenu.add("Parts of Surface Plotter");
    helpmenu.add("Maus Operationen");
    helpmenu.addSeparator();
    helpmenu.add("System Informationen");
    helpmenu.addSeparator();
    helpmenu.add("ber");
    
    mb.add(helpmenu);
    mb.setHelpMenu(helpmenu);       

    return mb;
  }

  /**
   * Gets the identifier of menu item.
   *
   * @return the identifier
   * @param item the menu item
   */
   
  private int whichMenu(MenuItem item) {
    for (int i=menubar.countMenus(); --i >= 0;) {
      Menu menu = menubar.getMenu(i);
      for (int j=menu.countItems(); --j >= 0;) {
        if (menu.getItem(j) == item) return (i+1)*100+j+1;
      }
    }
    return -1; 
  }

  /**
   * Gets the menu item given the identifier 
   *
   * @return the menu item
   * @param identifier the identifier
   */
   
  private MenuItem getMenuItem(int identifier) {
    int x,y;
    
    x = identifier / 100 - 1;
    y = identifier % 100 - 1; 
    
    Menu menu = menubar.getMenu(x);
    return menu.getItem(y);
  }

  /**
   * Displays help file.
   */
   
  private SurfaceHelp helpwindow = null;
  
  private void displayHelp(String helpfile) {
                                                                                                
    helpfile = "../" + helpfile;

    if (applet != null) {
      URL url = applet.getCodeBase();
     
      if (url != null) {
        try {
          url = new URL(url,helpfile);
        }
        catch (MalformedURLException e) {
          return;
        }
        applet.getAppletContext().showDocument(url,"help_window");
        return;
      }
    }
    
    if (helpwindow == null) {
      helpwindow = new SurfaceHelp(helpfile,applet);
      helpwindow.show();
    }
    else 
      helpwindow.show(helpfile);
  }

  /**
   * Processes menu events
   *
   * @param item the selected menu item
   */
   
  private void processMenuEvent(MenuItem item) {
    switch (whichMenu(item)) {

      case FILE_EXIT:       if (applet == null) {
                              dispose();
                              System.exit(0);
                            }
                            else
                              applet.disposeWindow();
                            break;
                           
      case OPT_HIDDEN:
      case OPT_SPECTRUM:
      case OPT_GRAYSCALE:
      case OPT_DUALSHADES:  for (int i=0; i < MODES_COUNT; i++)
                              ((CheckboxMenuItem)
                                getMenuItem(OPT_SHADES+i)).setState(false);
                            ((CheckboxMenuItem)item).setState(true);

                            // Warning. Falls through 
      case OPT_BOXED:
      case OPT_MESH:
      case OPT_FACEGRIDS:
      case OPT_SCALE:
      case OPT_XYTICKS:
      case OPT_ZTICKS:      canvas.destroyImage(); // redraws
                            canvas.repaint(); 
                            break;
      //case HELP_MENU:       displayHelp("help/menu.html");
      //                      break;      
      case HELP_FUNCTION:   displayHelp("help/function.html");
                            break;       
      //case HELP_PARTS:      displayHelp("help/parts.html");
      //                      break;       
      case HELP_MOUSE:      displayHelp("help/mouse.html");
                            break;
      case HELP_SYSTEMINFO: new SurfaceSysInfo(this);
                            break;
      case HELP_ABOUT:      new SurfaceAbout(this);
                            break;
      default:              break;
    }
  }
    
  /**
   * Allocates Memory
   */

  private void allocateMemory(int total) {
    vertex = null;
   
    // Releases memory being used    
    canvas.setValuesArray(null);
          
    try {
      vertex = new SurfaceVertex[total];
    }
    catch(OutOfMemoryError e) {
      function_panel.setErrorMessage("Zu wenig Hauptspeicher");
    }
    catch(Exception e) {
      function_panel.setErrorMessage("Fehler: " + e.toString());
    }
  }

  // avoids garbage collected.

  private final int FILE_EXIT       = 101;

  private final int OPT_BOXED       = 201;
  private final int OPT_MESH        = 202;
  private final int OPT_FACEGRIDS   = 203;
  private final int OPT_SCALE       = 204;
  private final int OPT_XYTICKS     = 206;
  private final int OPT_ZTICKS      = 207;

  private final int OPT_SHADES      = 209;
  private final int OPT_HIDDEN      = 209;
  private final int OPT_SPECTRUM    = 210;
  private final int OPT_GRAYSCALE   = 211;
  private final int OPT_DUALSHADES  = 212;

  //private final int HELP_MENU       = 301;
  private final int HELP_FUNCTION   = 301;
  //private final int HELP_PARTS      = 303;
  private final int HELP_MOUSE      = 302;

  private final int HELP_SYSTEMINFO = 304;
  private final int HELP_ABOUT      = 306;
}
