/*----------------------------------------------------------------------------------------*
 * SurfaceCanvas.java                                                                     *
 *                                                                                        *
 * Surface Plotter   version 1.10    14 Oct 1996                                          *
 *                   version 1.20     8 Nov 1996                                          *
 *                   version 1.30b1  17 May 1997                                          *
 *                   version 1.30b2  18 Oct 2001                                          *
 *                                                                                        *
 * Copyright (c) Yanto Suryono <yanto@fedu.uec.ac.jp>                                     *
 *                                                                                        *
 *                                                                                        *
 * MathOptAnimator   version 1.00     10 Feb 2002                                         *
 * Copyright (c) 2003 Ramazan Buzdemir <buzdemir@zpr.uni-koeln.de>                        *
 *                                                                                        *
 *                                                                                        *
 * This program is free software; you can redistribute it and/or modify it                *
 * under the terms of the GNU General Public License as published by the                  *
 * Free Software Foundation; either version 2 of the License, or (at your option)         *
 * any later version.                                                                     *
 *                                                                                        *
 * This program is distributed in the hope that it will be useful, but                    *
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or          *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for               *
 * more details.                                                                          *
 *                                                                                        *
 * You should have received a copy of the GNU General Public License along                *
 * with this program; if not, write to the Free Software Foundation, Inc.,                *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA                                  *
 *                                                                                        *
 *----------------------------------------------------------------------------------------*/

import java.awt.*;

/**
 * The class <code>SurfaceCanvas</code> is responsible
 * for the generation of surface images and user mouse events handling. 
 *
 * @author  Yanto Suryono
 */

public final class SurfaceCanvas extends Canvas {
  private SurfaceFrame frame;              // the parent, Surface Plotter frame
  
  private Image Buffer;                    // the backing buffer
  private Graphics BufferGC;               // the graphics context of backing buffer
  public  Projector projector;             // the projector
  private SurfaceVertex[] vertex;          // vertices array
  private int prevwidth, prevheight;       // canvas size  
  private float color;                     // color of surface
  private SurfaceVertex cop;               // center of projection
  
  private boolean image_drawn;             // image drawn flag
  private boolean data_available;          // data availability flag
  
  private float zfactor;
  
  // setting variables
  
  private int plot_mode;
  private int calc_divisions; 
  private boolean isBoxed, isMesh, isScaleBox, 
                  isDisplayXY, isDisplayZ, isDisplayGrids;
  private float xmin,xmax,ymin;
  private float ymax,zmin,zmax;

  // constants
                
  private static final int NORENDER    = 0;
  private static final int SPECTRUM    = 1;
  private static final int GRAYSCALE   = 2;
  private static final int DUALSHADE   = 3;
  
  private static final int TOP         = 0;
  private static final int CENTER      = 1;

  // for splitting polygons
  
  private static final int UPPER       =  1;
  private static final int COINCIDE    =  0;
  private static final int LOWER       = -1;
    
  /**
   * The constructor of <code>SurfaceCanvas</code>
   *
   * @param frame The parent frame. The frame must be an instance of 
   *              <code>SurfaceFrame</code>.                             
   * @see         SurfaceFrame
   */

  SurfaceCanvas(SurfaceFrame frame) {
    this.frame = frame;
    Buffer = null;
    BufferGC = null;
    image_drawn = false;
    data_available = false;
    prevwidth = prevheight = -1;
    projector = new Projector();
    projector.setDistance(70);
    if (getToolkit().getScreenSize().width < 1024)
      projector.set2DScaling(10);
    else
      projector.set2DScaling(14);
    projector.setRotationAngle(125);
    projector.setElevationAngle(10);
  }
  
  /**
   * Destroys the internal image. It will force <code>SurfaceCanvas</code>
   * to regenerate all images when the <code>paint</code> method is called. 
   */

  public void destroyImage() {
    image_drawn = false;
  }

  /**
   * Sets the x and y ranges of calculated surface vertices. 
   * The ranges will not affect surface appearance. They affect axes 
   * scale appearance. 
   *
   * @param xmin the minimum x
   * @param xmax the maximum x
   * @param ymin the minimum y
   * @param ymax the maximum y
   */

  public void setRanges(float xmin, float xmax, float ymin, float ymax) {
    this.xmin = xmin;
    this.xmax = xmax;
    this.ymin = ymin;
    this.ymax = ymax;
  }
  
  /**
   * Gets the current x, y, and z ranges. 
   *
   * @return array of x,y, and z ranges in order of 
   *         xmin, xmax, ymin, ymax, zmin, zmax 
   */

  public float[] getRanges() {
    float[] ranges = new float[6];
    
    ranges[0] = xmin;
    ranges[1] = xmax;
    ranges[2] = ymin;
    ranges[3] = ymax;
    ranges[4] = zmin;
    ranges[5] = zmax;
    
    return ranges;
  }
    
  /**
   * Sets the data availability flag. If this flag is <code>false</code>,
   * <code>SurfaceCanvas</code> will not generate any surface image, even
   * if the data is available. But it is the programmer's responsiblity 
   * to set this flag to <code>false</code> when data is not available. 
   *
   * @param avail the availability flag
   */

  public void setDataAvailability(boolean avail) {
    data_available = avail;
    is_data_available = avail;  // see Handlers for mouse input events section
  }

  /**
   * Sets the new vertices array of surface. 
   *
   * @param vertex the new vertices array   
   * @see   #getValuesArray
   */

  public void setValuesArray(SurfaceVertex[] vertex) {
    this.vertex = vertex;
  }
  
  /**
   * Gets the current vertices array.
   *
   * @return current vertices array
   * @see    #setValuesArray
   */

  public SurfaceVertex[] getValuesArray() {
    if (!data_available) return null;
    return vertex;
  }
    
  private boolean is_data_available;   // holds the original data availability flag
  private boolean dragged;             // dragged flag
  private int     click_x, click_y;    // previous mouse cursor position
    
  /**
   * <code>mouseDown</code> event handler. Sets internal tracking variables
   * for dragging operations.
   *
   * @param e the event
   * @param x the x coordinate of cursor
   * @param y the y coordinate of cursor
   */

  public boolean mouseDown(Event e, int x, int y) { 	
    click_x = x;
    click_y = y;
    return true;
  }
  
  /**
   * <code>mouseUp<code> event handler. Regenerates image if dragging operations 
   * have been done with the delay regeneration flag set on.
   *
   * @param e the event
   * @param x the x coordinate of cursor
   * @param y the y coordinate of cursor
   */

  public boolean mouseUp(Event e, int x, int y) {  	
    if (frame.setting_panel.isExpectDelay() && dragged) {	
      destroyImage(); 
      data_available = is_data_available;
      repaint(); 
      dragged = false;
    }
    return true; 
  }

  /**
   * <code>mouseDrag<code> event handler. Tracks dragging operations. 
   * Checks the delay regeneration flag and does proper actions.
   *
   * @param e the event
   * @param x the x coordinate of cursor
   * @param y the y coordinate of cursor
   */

  public boolean mouseDrag(Event e, int x, int y) { 	
    float new_value = 0.0f;
    
    if (true) {
      if (e.controlDown()) {
        projector.set2D_xTranslation(
        projector.get2D_xTranslation() + (x - click_x));
        projector.set2D_yTranslation(
        projector.get2D_yTranslation() + (y - click_y));
      }
      else
      if (e.shiftDown()) {
        new_value = projector.get2DScaling() + (y - click_y) * 0.5f;
        if (new_value > 100000.0f) new_value = 100000.0f;
        if (new_value < 2.0f) new_value = 2.0f;
        projector.set2DScaling(new_value);
      }
      else {
        new_value = projector.getRotationAngle() + (x - click_x);
        while (new_value > 360) new_value -= 360;
        while (new_value < 0) new_value += 360;
        projector.setRotationAngle(new_value);
        new_value = projector.getElevationAngle() + (y - click_y);
        if (new_value > 90) new_value = 90;
          else
        if (new_value < -90) new_value = -90;
        projector.setElevationAngle(new_value);
      } 
      image_drawn = false;
      if (!frame.setting_panel.isExpectDelay()) {
        repaint(); 
      }
      else {
        if (!dragged) {
          is_data_available = data_available;
          dragged = true;
        }
        data_available = false;
        repaint();
      }        
    }
    click_x = x;
    click_y = y;
    return true;
  }

  /**
   * Paints surface. Creates surface plot, based on current vertices array, 
   * contour plot flag, and density plot flag.
   * If no data is available, creates image of base plane and axes. 
   *
   * @param g the graphics context to paint
   * @see   #setValuesArray
   * @see   #setDataAvailability   
   */

  public void paint(Graphics g) {
    if ((bounds().width <= 0) || (bounds().height <= 0)) return;

    // backing buffer creation

    if ((bounds().width != prevwidth) || 
        (bounds().height != prevheight) || 
        (BufferGC == null)) {
      frame.function_panel.setMessage("Neue Bildgrsse: " + bounds().width + "x" + bounds().height);
      projector.setProjectionArea(new Rectangle(0,0,bounds().width, bounds().height));  
      image_drawn = false;
      if (Buffer != null) Buffer.flush();
      Buffer = createImage(bounds().width, bounds().height);
      if (BufferGC != null) BufferGC.dispose();
      BufferGC = Buffer.getGraphics();     
      prevwidth = bounds().width;
      prevheight = bounds().height;
    }

    importVariables();

    if (image_drawn && (Buffer != null))
      g.drawImage(Buffer,0,0,this);
    else
      if (data_available)
        plot();
      else {
        g.setColor(Color.lightGray);
        g.fillRect(0,0,bounds().width,bounds().height);
        drawBoxGridsTicksLabels(g,true);
      }
  }

  /**
   * Updates image. Just call the <code>paint</code> method to
   * avoid flickers.
   *
   * @param g the graphics context to update
   * @see   #paint
   */

  public void update(Graphics g) {
    paint(g);                        // do not erase, just paint
  }  

  /**
   * The implementation of <code>Runnable</code> interface.
   * Performs surface plotting.
   */

  public void plot() {


    int fontsize = (int)(Math.round(projector.get2DScaling() * 0.5));
    BufferGC.setFont(new Font("Helvetica",Font.PLAIN,fontsize));
      
    // surface plot
    plotSurface();
      
    // force repaint NOW
    Graphics g = getGraphics();
    paint(g);
    g.dispose();
    
  } 
  
  /**
   * Returns the preferred size of this object. This will be the initial
   * size of <code>SurfaceCanvas</code>.
   *
   * @return the preferred size.
   */

  public Dimension preferredSize() {
    if (getToolkit().getScreenSize().width < 1024)
     return new Dimension(550,290);
    else
     return new Dimension(550,410);
  }

/*----------------------------------------------------------------------------------------*
 *                            Private methods begin here                                  *
 *----------------------------------------------------------------------------------------*/

  private int factor_x, factor_y;   // conversion factors 
  private int t_x, t_y, t_z;        // determines ticks density
  Point tickpos;  
  boolean  x_left = false, y_left = false;

  /**
   * Draws the bounding box of surface.
   */

  private final void drawBoundingBox() {  	
    Point startingpoint, projection;
    
    startingpoint = projector.project(factor_x*10,factor_y*10,10);  
    BufferGC.setColor(Color.black);
    projection =  projector.project(-factor_x*10,factor_y*10,10);
    if (projector.project(-factor_x*10,-factor_y*10,10).y < 
        projector.project(factor_x*10,factor_y*10,10).y)
        BufferGC.drawLine(startingpoint.x, startingpoint.y, projection.x, projection.y); 
    projection =  projector.project(factor_x*10,-factor_y*10,10);  
    BufferGC.drawLine(startingpoint.x, startingpoint.y, projection.x, projection.y);
    projection =  projector.project(factor_x*10,factor_y*10,-10);  
    BufferGC.drawLine(startingpoint.x, startingpoint.y, projection.x, projection.y);
  }

  /**
   * Draws the base plane. The base plane is the x-y plane.
   *
   * @param g the graphics context to draw.
   * @param x used to retrieve x coordinates of drawn plane from this method. 
   * @param y used to retrieve y coordinates of drawn plane from this method.
   */

  private final void drawBase(Graphics g, int[] x, int[] y) {
    Point projection = projector.project(-10,-10,-10);                  
    x[0] = projection.x; y[0] = projection.y;
    projection = projector.project(-10, 10,-10);                  
    x[1] = projection.x; y[1] = projection.y;
    projection = projector.project( 10, 10,-10);                  
    x[2] = projection.x; y[2] = projection.y; 
    projection = projector.project( 10,-10,-10);                  
    x[3] = projection.x; y[3] = projection.y;
    x[4] = x[0]; y[4] = y[0];

    if (plot_mode == NORENDER)
      g.setColor(Color.lightGray);
    else     
      g.setColor(new Color(192,220,192));      
      g.fillPolygon(x,y,4);
    g.setColor(Color.black);
    g.drawPolygon(x,y,5);
  }
  
  private final void xyGrids(Graphics g) {
    int px, py;
    for (int i=-9; i <= 9; i++) {
      g.setColor(Color.black);       
      projection = projector.project(factor_x*10,factor_y*10,-10);
      px = projection.x; py = projection.y;
      projection = projector.project(-factor_x*10,factor_y*10,-10);
      g.drawLine(px,py,projection.x,projection.y); 
      projection = projector.project(factor_x*10,-factor_y*10,-10);
      g.drawLine(px,py,projection.x,projection.y);
                   
      if (isDisplayXY || isDisplayGrids) {
        if (!isDisplayGrids || (i % (t_y / 2) == 0) || isDisplayXY) {
          if (isDisplayGrids && (i % t_y == 0)) 
            projection = projector.project(-factor_x * 10,i,-10);
          else {
            if (i % t_y != 0) 
              projection = projector.project(factor_x * 9.8f,i,-10);
            else
              projection = projector.project(factor_x * 9.5f,i,-10);
          }
          tickpos = projector.project(factor_x * 10,i,-10);
          g.drawLine(projection.x,projection.y,tickpos.x,tickpos.y);
          if ((i % t_y == 0) && isDisplayXY) {
            tickpos = projector.project(factor_x * 10.5f,i,-10);
            //y-axis
            if (y_left)
              outFloat(g,tickpos.x,tickpos.y,
                       (float)((double)(i+10)/20*(ymax-ymin)+ymin),
                       Label.LEFT,TOP);
            else
              outFloat(g,tickpos.x,tickpos.y,
                       (float)((double)(i+10)/20*(ymax-ymin)+ymin),
                       Label.RIGHT,TOP);
          }
        }  
        if (!isDisplayGrids || (i % (t_x / 2) == 0) || isDisplayXY) {
          if (isDisplayGrids && (i % t_x == 0))
            projection = projector.project(i,-factor_y * 10,-10);
          else {
            if (i % t_x != 0) 
              projection = projector.project(i,factor_y * 9.8f,-10);
            else
              projection = projector.project(i,factor_y * 9.5f,-10);
          }
          tickpos = projector.project(i,factor_y * 10,-10);
          g.drawLine(projection.x,projection.y,tickpos.x,tickpos.y);
          if ((i % t_x == 0) && isDisplayXY) {
            tickpos = projector.project(i,factor_y * 10.5f,-10);
            //x-axis
            if (x_left)
              outFloat(g,tickpos.x,tickpos.y,
                       (float)((double)(i+10)/20*(xmax-xmin)+xmin),
                       Label.LEFT,TOP);
            else
              outFloat(g,tickpos.x,tickpos.y,
                       (float)((double)(i+10)/20*(xmax-xmin)+xmin),
                       Label.RIGHT,TOP);
          }
        }
      }  
      
      if (isDisplayXY) {
        tickpos = projector.project(0,factor_y * 14,-10);
        outString(g,tickpos.x,tickpos.y,"X",Label.CENTER,TOP);
        tickpos = projector.project(factor_x * 14,0,-10);
        outString(g,tickpos.x,tickpos.y,"Y",Label.CENTER,TOP);
      }
    }
  }

  /**
   * Draws non-surface parts, i.e: bounding box, axis grids, axis ticks,
   * axis labels, base plane. 
   *
   * @param g         the graphics context to draw
   * @param draw_axes if <code>true</code>, only draws base plane and z axis
   */

  private final void drawBoxGridsTicksLabels(Graphics g, boolean draw_axes) {
    Point    projection;   
    boolean  bool1 = false, bool2 = false; 
    int      x[],y[],px,py;

    x = new int[5]; y = new int[5];
    if (projector == null) return;

    if (draw_axes) {
      drawBase(g,x,y);
      projection = projector.project(0,0,-10);                  
      x[0] = projection.x; y[0] = projection.y;
      projection = projector.project(10.5f,0,-10);                  
      g.drawLine(x[0],y[0],projection.x,projection.y);
      if (projection.x < x[0]) 
        outString(g,(int)(1.05*(projection.x-x[0]))+x[0],
                    (int)(1.05*(projection.y-y[0]))+y[0],"x",Label.RIGHT,TOP);
      else
        outString(g,(int)(1.05*(projection.x-x[0]))+x[0],
                    (int)(1.05*(projection.y-y[0]))+y[0],"x",Label.LEFT,TOP);
      projection = projector.project(0,11.5f,-10);                  
      g.drawLine(x[0],y[0],projection.x,projection.y);
      if (projection.x < x[0]) 
        outString(g,(int)(1.05*(projection.x-x[0]))+x[0],
                    (int)(1.05*(projection.y-y[0]))+y[0],"y",Label.RIGHT,TOP);
      else
        outString(g,(int)(1.05*(projection.x-x[0]))+x[0],
                     (int)(1.05*(projection.y-y[0]))+y[0],"y",Label.LEFT,TOP);
      projection = projector.project(0,0,10.5f);                  
      g.drawLine(x[0],y[0],projection.x,projection.y);
      outString(g,(int)(1.05*(projection.x-x[0]))+x[0],                    
                  (int)(1.05*(projection.y-y[0]))+y[0],"z",Label.CENTER,CENTER);
    } 
    else {
      projection = projector.project(10.5f,0,-10);                  
      px = projection.x;
      py = projection.y;      
      projection = projector.project(-10.5f,0,-10);
      bool1 = projection.x > px;
           
      projection = projector.project(0,10.5f,-10);  
      px = projection.x;
      py = projection.y;                        
      projection = projector.project(0,-10.5f,-10);    
      bool2 = projection.x > px;
      
      if (bool1)
       if (bool2) {
      	factor_x = -1;
      	factor_y = 1;
      	y_left = true;
      	x_left = false;
       }
       else {
        factor_x = 1;
      	factor_y = 1;
       	y_left = false;
      	x_left = true;     	
       }
      else
       if (bool2) {
      	factor_x = -1;
      	factor_y = -1;
      	y_left = false;
      	x_left = true;      	
       }
       else {
        factor_x = 1;
      	factor_y = -1;
      	y_left = true;
      	x_left = false;      	
       }      
       
      setAxesScale(); drawBase(g,x,y);

      if (isBoxed) {
        projection = projector.project(factor_x*10,factor_y*10,10);
        px = projection.x;
        py = projection.y;
        projection =  projector.project(-factor_x*10,factor_y*10,10);
        if (!(projector.project(-factor_x*10,-factor_y*10,10).y < 
            projector.project(factor_x*10,factor_y*10,10).y))
            g.drawLine(px, py, projection.x, projection.y);      	
      	
        projection = projector.project(-factor_x*10,-factor_y*10,-10);
        x[0] = projection.x; y[0] = projection.y;
        projection = projector.project(-factor_x*10,-factor_y*10,10);                  
        x[1] = projection.x; y[1] = projection.y;
        projection = projector.project(factor_x*10,-factor_y*10,10);
        x[2] = projection.x; y[2] = projection.y;
        projection = projector.project(factor_x*10,-factor_y*10,-10);
        x[3] = projection.x; y[3] = projection.y;
        x[4] = x[0]; y[4] = y[0];
  
        if (plot_mode == NORENDER)
          g.setColor(Color.lightGray);
        else
          g.setColor(new Color(192,220,192));                    
        g.fillPolygon(x,y,4);
        g.setColor(Color.black);
        g.drawPolygon(x,y,5);

        projection = projector.project(-factor_x*10,factor_y*10,10);
        x[2] = projection.x; y[2] = projection.y;
        projection = projector.project(-factor_x*10,factor_y*10,-10);
        x[3] = projection.x; y[3] = projection.y;
        x[4] = x[0]; y[4] = y[0];
  
        if (plot_mode == NORENDER)
          g.setColor(Color.lightGray);
        else
          g.setColor(new Color(192,220,192));         
        g.fillPolygon(x,y,4);
        g.setColor(Color.black);
        g.drawPolygon(x,y,5);
      }
      else
      if (isDisplayZ) {
        projection = projector.project(factor_x*10,-factor_y*10,-10);
        x[0] = projection.x; y[0] = projection.y;  
        projection = projector.project(factor_x*10,-factor_y*10,10);
        g.drawLine(x[0],y[0],projection.x,projection.y);

        projection = projector.project(-factor_x*10,factor_y*10,-10);
        x[0] = projection.x; y[0] = projection.y;  
        projection = projector.project(-factor_x*10,factor_y*10,10);
        g.drawLine(x[0],y[0],projection.x,projection.y);
      }
      
      if (projector.project(-factor_x*10,-factor_y*10,-10).y < 
          projector.project(factor_x*10,factor_y*10,-10).y) xyGrids(g);
      
      for (int i=-9; i <= 9; i++) {
        // z grids and ticks

        if (isDisplayZ || (isDisplayGrids && isBoxed)) {
          if (!isDisplayGrids || (i % (t_z / 2) == 0) || isDisplayZ) {
            if (isBoxed && isDisplayGrids && (i % t_z == 0)) {
              projection = projector.project(-factor_x*10,-factor_y*10,i);
              tickpos = projector.project(-factor_x*10,factor_y*10,i);
            }
            else {
              if (i % t_z == 0) 
                projection = projector.project(-factor_x*10,factor_y*9.5f,i);
              else
                projection = projector.project(-factor_x*10,factor_y*9.8f,i);  
              tickpos = projector.project(-factor_x*10,factor_y*10,i);
            }                            
            g.drawLine(projection.x,projection.y,tickpos.x,tickpos.y);
            if (isDisplayZ) {
              tickpos = projector.project(-factor_x*10,factor_y*10.5f,i);
              if (i % t_z == 0) {
              	//z-axis1
                if (x_left)
                  outFloat(g,tickpos.x,tickpos.y,
                           (float)((double)(i+10)/20*(zmax-zmin)+zmin),
                           Label.LEFT,CENTER);
                else
                  outFloat(g,tickpos.x,tickpos.y,
                           (float)((double)(i+10)/20*(zmax-zmin)+zmin),
                           Label.RIGHT,CENTER);
              }
            }
            if (isDisplayGrids && isBoxed && (i % t_z == 0)) {
              projection = projector.project(-factor_x*10,-factor_y*10,i);
              tickpos = projector.project(factor_x*10,-factor_y*10,i);
            } 
            else {
              if (i % t_z == 0)
                projection = projector.project(factor_x*9.5f,-factor_y*10,i);
              else
                projection = projector.project(factor_x*9.8f,-factor_y*10,i);
              tickpos = projector.project(factor_x*10,-factor_y*10,i);
            }             
            g.drawLine(projection.x,projection.y,tickpos.x,tickpos.y);
            if (isDisplayZ) {
              tickpos = projector.project(factor_x*10.5f,-factor_y*10,i);
              if (i % t_z == 0) {
              	//z-axis2
                if (y_left)
                  outFloat(g,tickpos.x,tickpos.y,
                           (float)((double)(i+10)/20*(zmax-zmin)+zmin),
                           Label.LEFT,CENTER);
                else
                  outFloat(g,tickpos.x,tickpos.y,
                           (float)((double)(i+10)/20*(zmax-zmin)+zmin),
                           Label.RIGHT,CENTER);
              }
            }
            if (isDisplayGrids && isBoxed) {
              if (i % t_y == 0) {
                projection = projector.project(-factor_x * 10,i,-10);
                tickpos = projector.project(-factor_x * 10,i,10);
                g.drawLine(projection.x,projection.y,tickpos.x,tickpos.y);
              }
              if (i % t_x  == 0) {
                projection = projector.project(i,-factor_y * 10,-10);
                tickpos = projector.project(i,-factor_y * 10,10);
                g.drawLine(projection.x,projection.y,tickpos.x,tickpos.y);
              }
            }   
          }
        }   
      }
    }
  }

  /**
   * Imports setting variables from parent frame. This method is before doing 
   * any plotting operations
   *
   * @see     #SurfaceVanvas (constructor)
   */

  private void importVariables() {
    plot_mode = frame.getPlotMode();
    isBoxed = frame.isBoxed();
    isMesh = frame.isMesh();
    isScaleBox = frame.isScaleBox();
    isDisplayXY = frame.isDisplayXY();
    isDisplayZ = frame.isDisplayZ();
    isDisplayGrids = frame.isDisplayGrids();
    calc_divisions = frame.setting_panel.getCalcDivisions();  
  }

  /**
   * Sets the axes scaling factor. Computes the proper axis lengths
   * based on the ratio of variable ranges. The axis lengths will
   * also affect the size of bounding box. 
   */

  private final void setAxesScale() {
    float scale_x, scale_y, scale_z ,divisor;
    int   longest;

    if (!isScaleBox) {
      projector.setScaling(1); 
      t_x = t_y = t_z = 4; 
      return;
    }

    scale_x = xmax - xmin; scale_y = ymax - ymin; scale_z = zmax - zmin;

    if (scale_x < scale_y) {
      if (scale_y < scale_z) {
        longest = 3; divisor = scale_z;
      }
      else {
        longest = 2; divisor = scale_y;
      }
    }
    else {
      if (scale_x < scale_z) {
        longest = 3; divisor = scale_z;
      }
      else {
        longest = 1; divisor = scale_x;
      }
    }
    scale_x /= divisor; scale_y /= divisor; scale_z /= divisor;
 
    if ((scale_x < 0.2f) || (scale_y < 0.2f) && (scale_z < 0.2f)) {
      switch(longest) {
        case 1: if (scale_y < scale_z) {
                  scale_y /= scale_z; scale_z = 1.0f;
                }
                else {
                  scale_z /= scale_y; scale_y = 1.0f;
                } break;
        case 2: if (scale_x < scale_z) {
                  scale_x /= scale_z; scale_z = 1.0f;
                }
                else {
                  scale_z /= scale_x; scale_x = 1.0f;
                } break;
        case 3: if (scale_y < scale_x) {
                  scale_y /= scale_x; scale_x = 1.0f;
                }
                else {
                  scale_x /= scale_y; scale_y = 1.0f;
                } break;
      }
    }
    if (scale_x < 0.2f) scale_x = 1.0f; projector.setXScaling(scale_x);
    if (scale_y < 0.2f) scale_y = 1.0f; projector.setYScaling(scale_y);
    if (scale_z < 0.2f) scale_z = 1.0f; projector.setZScaling(scale_z);

    if (scale_x < 0.5f) t_x = 8; else t_x = 4;
    if (scale_y < 0.5f) t_y = 8; else t_y = 4;
    if (scale_z < 0.5f) t_z = 8; else t_z = 4;
  }

  /**
   * Draws string at the specified coordinates with the specified alignment.
   *
   * @param g       graphics context to draw
   * @param x       the x coordinate
   * @param y       the y coordinate
   * @param s       the string to draw
   * @param x_align the alignment in x direction
   * @param y_align the alignment in y direction    
   */

  private final void outString(Graphics g, int x, int y, 
                               String s, int x_align, int y_align) {
    switch (y_align) {
      case TOP    : y += g.getFontMetrics(g.getFont()).getAscent(); break;
      case CENTER : y += g.getFontMetrics(g.getFont()).getAscent()/2; break;
    }
    switch (x_align) {
      case Label.LEFT  : g.drawString(s,x,y); 
                         break;
      case Label.RIGHT : g.drawString(s,x-g.getFontMetrics(
                                      g.getFont()).stringWidth(s),y); 
                         break;
      case Label.CENTER: g.drawString(s,x-g.getFontMetrics(
                                      g.getFont()).stringWidth(s)/2,y); 
                         break;
    }     
  }

  /**
   * Draws float at the specified coordinates with the specified alignment.
   *
   * @param g       graphics context to draw
   * @param x       the x coordinate
   * @param y       the y coordinate
   * @param f       the float to draw
   * @param x_align the alignment in x direction
   * @param y_align the alignment in y direction    
   */

  private final void outFloat(Graphics g, int x, int y, 
                              float f, int x_align, int y_align) {
    String s = Float.toString(f);
    outString(g,x,y,s,x_align,y_align);
  }

/*----------------------------------------------------------------------------------------*
 *                       Plotting routines and methods begin here                         *
 *----------------------------------------------------------------------------------------*/

  private float color_factor;
  private Point projection;   
  private Color line_color; 
  
  private int poly_x[] = new int[9];
  private int poly_y[] = new int[9];
  
  private float polynom_x[] = new float[9];
  private float polynom_y[] = new float[9];
  private float polynom_z[] = new float[9];

  /**
   * Plots a single plane
   *
   * @param vertex vertices array of the plane
   * @param verticescount number of vertices to process 
   */

  private final void plotPlane(SurfaceVertex[] vertex, int verticescount) {
    int     count,loop,index;
    float   z,result;
    boolean low1,low2;
    boolean valid1,valid2;

    if (verticescount < 3) return;
    count = 0; z = 0.0f; line_color = Color.black;
    low1 = (vertex[0].z < zmin); valid1 = !low1 && (vertex[0].z <= zmax); index = 1;
    for (loop = 0; loop < verticescount; loop++) {
      low2 = (vertex[index].z < zmin); valid2 = !low2 && (vertex[index].z <= zmax);
      if ((valid1 || valid2) || (low1^low2)) {
        if (!valid1) {
          if (low1) result = zmin; else result = zmax;
          float ratio = (result-vertex[index].z)/(vertex[loop].z-vertex[index].z);
          float new_x = ratio*(vertex[loop].x-vertex[index].x)+vertex[index].x; 
          float new_y = ratio*(vertex[loop].y-vertex[index].y)+vertex[index].y;
          
          polynom_x[count] = new_x;
          polynom_y[count] = new_y;
          if (low1) {
            projection = projector.project(new_x,new_y,-10);
            polynom_z[count] = zmin;
          }
          else {
            projection = projector.project(new_x,new_y,10);
            polynom_z[count] = zmax;
          }
          
          poly_x[count] = projection.x;
          poly_y[count] = projection.y;
           
          count++; z += result;
        } 
        if (valid2) {	
          projection = projector.project(vertex[index].x, 
                                         vertex[index].y, 
                                         (vertex[index].z-zmin)*zfactor-10);
                                         
          polynom_x[count] = vertex[index].x;
          polynom_y[count] = vertex[index].y;
          polynom_z[count] = vertex[index].z;
          
          poly_x[count] = projection.x;
          poly_y[count] = projection.y;
          
          count++; z += vertex[index].z;
        }
        else {
          if (low2) result = zmin; else result = zmax;
          float ratio = (result-vertex[loop].z)/(vertex[index].z-vertex[loop].z);
          float new_x = ratio*(vertex[index].x-vertex[loop].x)+vertex[loop].x; 
          float new_y = ratio*(vertex[index].y-vertex[loop].y)+vertex[loop].y;
          
          
          polynom_x[count] = new_x;
          polynom_y[count] = new_y;
          if (low2) {
            projection = projector.project(new_x,new_y,-10);
            polynom_z[count] = zmin;
          }
          else {
            projection = projector.project(new_x,new_y,10);
            polynom_z[count] = zmax;
          }
          
          poly_x[count] = projection.x;
          poly_y[count] = projection.y;
          
          count++; z += result;
        } 
      }
      if (++index == verticescount) index = 0; 
      valid1 = valid2; low1 = low2;
    }
    if (count > 0) {

      switch (plot_mode) {
        case NORENDER:    BufferGC.setColor(Color.lightGray); 
                          break;
        case SPECTRUM:    z = 0.8f-(z/count-zmin)*color_factor;
                          BufferGC.setColor(Color.getHSBColor(z,1.0f,1.0f)); 
                          break;
        case GRAYSCALE:   z = (z/count-zmin)*color_factor;
                          BufferGC.setColor(Color.getHSBColor(0,0,z));
                          if (z < 0.3f) line_color = new Color(0.6f,0.6f,0.6f);                    
                          break; 
        case DUALSHADE:   z = (z/count-zmin)*color_factor+0.4f;
                          BufferGC.setColor(Color.getHSBColor(color,0.7f,z));
                          break;
      }
                               
      BufferGC.fillPolygon(poly_x,poly_y,count);
      BufferGC.setColor(line_color);
      if (isMesh || (plot_mode == NORENDER)) {
      	
        polynom_x[count] = polynom_x[0];
        polynom_y[count] = polynom_y[0];
        polynom_z[count] = polynom_z[0];
      	
        poly_x[count] = poly_x[0];
        poly_y[count] = poly_y[0];
                
        count++;
        BufferGC.drawPolygon(poly_x,poly_y,count);
        drawLineInPolygon(vertex, count);
      }
      
    }
    
  }

  private final void drawLineInPolygon(SurfaceVertex[] vertex, int count) {
    
    if (frame.algorithm_panel.algorithm.points_x == null) return;
    
    float xfactor = 20/(xmax-xmin);
    float yfactor = 20/(ymax-ymin);
        
    float x1 = 0.0f, y1 = 0.0f, z1 = 0.0f;   // Gerade 1
    float x2 = 0.0f, y2 = 0.0f, z2 = 0.0f;
    float m1 = 0.0f, b1 = 0.0f;

    float u1 = 0.0f, v1 = 0.0f, w1 = 0.0f;   // Gerade 2
    float u2 = 0.0f, v2 = 0.0f, w2 = 0.0f;
    float m2 = 0.0f, b2 = 0.0f;
        
    float s0 = 0.0f, t0 = 0.0f, h0 = 0.0f;   // Schnittpunkt 0
    float s1 = 0.0f, t1 = 0.0f, h1 = 0.0f;   // Schnittpunkt 1
    float s2 = 0.0f, t2 = 0.0f, h2 = 0.0f;   // Schnittpunkt 2
    
    float l1 = 0.0f, l2 = 0.0f;              // Lambda
    int anzsp = 0;                           // Anzahl Schnittpunkte Gerade 1 und Polygon
    
    int anzsp1 = 0;                          // Anzahl Schnittpunkte (x1,y1)+ und Polygon
    int anzsp2 = 0;                          // Anzahl Schnittpunkte (x2,y2)+ und Polygon

    for (int j = 0; j <= frame.algorithm_panel.algorithm.k; j++) {
      
      if (j%2 == 0)
        BufferGC.setColor(Color.red);
      else
        BufferGC.setColor(Color.blue);

      x1 = (frame.algorithm_panel.algorithm.points_x[j]-xmin)*xfactor-10;
      y1 = (frame.algorithm_panel.algorithm.points_y[j]-ymin)*yfactor-10;
      anzsp1 = 0;

      if (j > 0) {
        x2 = (frame.algorithm_panel.algorithm.points_x[j-1]-xmin)*xfactor-10;
        y2 = (frame.algorithm_panel.algorithm.points_y[j-1]-ymin)*yfactor-10;

        anzsp = 0;
        anzsp2 = 0;
      }

      for (int k = 1; k < count; k++) {
      
        u1 = polynom_x[k];
        v1 = polynom_y[k];
        w1 = polynom_z[k];
        
        u2 = polynom_x[k-1];
        v2 = polynom_y[k-1];
        w2 = polynom_z[k-1];
        
        
        if (u1 != u2) {
          l1 = (x1 - u1) / (u2 - u1);
          if ( (0.0f <= l1) && (l1 <= 1.0f) ) anzsp1++;
        }
        
        if (v1 != v2) {
          l1 = (y1 - v1) / (v2 - v1);
          if ( (0.0f <= l1) && (l1 <= 1.0f) ) anzsp1++;
        }
        
        if (j > 0) {
        	
          if (u1 != u2) {
            l1 = (x2 - u1) / (u2 - u1);
            if ( (0.0f <= l1) && (l1 <= 1.0f) ) anzsp2++;
          }
        
          if (v1 != v2) {
            l1 = (y2 - v1) / (v2 - v1);
            if ( (0.0f <= l1) && (l1 <= 1.0f) ) anzsp2++;
          }
          
          if ( (x1 == x2) && (u1 == u2) ) {
            l1 = -1.0f; l2 = -1.0f;
          }
          else {
            if ( (x1 == x2) && (u1 != u2) ) {
              m2 = (v2-v1) / (u2-u1);
              b2 = v1 - m2*u1;
              
              s0 = x1; t0 = m2 * x1 + b2;  // Schnittpunkt (s0,t0)
              
              l1 = (t0 - y1) / (y2 - y1);  // Gerade 1 = t0
              l2 = (s0 - u1) / (u2 - u1);  // Gerade 2 = s0
            }
            else {
              if ( (x1 != x2) && (u1 == u2) ) {
            	m1 = (y2-y1) / (x2-x1);
            	b1 = y1 - m1*x1;
            	
            	s0 = u1; t0 = m1 * u1 + b1;  // Schnittpunkt (s0,t0)
              	  
              	l1 = (s0 - x1) / (x2 - x1);  // Gerade 1 = s0
              	l2 = (t0 - v1) / (v2 - v1);  // Gerade 2 = t0
              }
              else {
                m1 = (y2-y1) / (x2-x1);
                b1 = y1 - m1*x1;
                m2 = (v2-v1) / (u2-u1);
                b2 = v1 - m2*u1;
                
                if (m1 == m2) {
                  l1 = -1.0f; l2 = -1.0f;
                }
                else {
                  s0 = (b2-b1) / (m1-m2); t0 = m1*s0 +b1; // Schnittpunkt (s0,t0)
                  l1 = (s0 - x1) / (x2 - x1);  // Gerade 1 = s0
                  l2 = (s0 - u1) / (u2 - u1);  // Gerade 2 = s0
                }
              }
            }
          }

          if ((0.0f <= l1) && (l1 <= 1.0f) && (0.0f <= l2) && (l2 <= 1.0f)) {
            h0 = w1 + l2*(w2-w1);
            if (anzsp == 0) {
              s1 = s0; t1 = t0; h1 = h0; anzsp = 1;
            }
            else {
              if ( (s1 != s0) || (t1 != t0) ) {
                s2 = s0; t2 = t0; h2 = h0; anzsp = 2;
              }
            }
          }
              
        } // if (j>0)
          
      } // for k
        
      if (anzsp1 > 3) {                          // (x1,y1) in Polygon
        z1 = eval_z(x1,y1,vertex);
        if (j == 0)
          DrawPoint(x1,y1,z1);
      }
        
      if (j > 0) {
      	
	if (anzsp2 > 3)                          // (x2,y2) in Polygon
          z2 = eval_z(x2,y2,vertex);
          
        if (anzsp == 0) {
          if ( (anzsp1 > 3) && (anzsp2 > 3) )
            DrawLine(x1,y1,z1,x2,y2,z2);
        }
        else {
          if (anzsp == 1) {
            if (anzsp1 > 3)
              DrawLine(x1,y1,z1,s1,t1,h1);
            if (anzsp2 > 3)
              DrawLine(x2,y2,z2,s1,t1,h1);
          }
          else {
            DrawLine(s1,t1,h1,s2,t2,h2);
          }
        }
       
      } // if (j > 0)
        
    }  // for j
      
  }
  
  private final float eval_z(float x, float y, SurfaceVertex[] vertex) {
    
    float l;
    float c0, c1, c2, c3;
    float z1, z2;

    l = (x-vertex[0].x) / (vertex[1].x-vertex[0].x);
    c0 = vertex[0].z + l*(vertex[1].z-vertex[0].z);
    
    l = (y-vertex[1].y) / (vertex[2].y-vertex[1].y);
    c1 = vertex[1].z + l*(vertex[2].z-vertex[1].z);
    
    l = (x-vertex[2].x) / (vertex[3].x-vertex[2].x);
    c2 = vertex[2].z + l*(vertex[3].z-vertex[2].z);
    
    l = (y-vertex[3].y) / (vertex[0].y-vertex[3].y);
    c3 = vertex[3].z + l*(vertex[0].z-vertex[3].z);
    
    
    l = (y-vertex[0].y) / (vertex[2].y-vertex[0].y);
    z1 = c0 + l*(c2-c0);
    
    l = (x-vertex[1].x) / (vertex[0].x-vertex[1].x);
    z2 = c1 + l*(c3-c1);
    
    return (z1+z2) / 2.0f;

  }
  
  private final void DrawLine(float x1, float y1, float z1, float x2, float y2, float z2) {
    
    SurfaceVertex vertex;
    Point p0, p1;
    
    vertex = new SurfaceVertex(x1, y1, z1);
    p0 = projector.project(vertex.x, vertex.y, (vertex.z-zmin)*zfactor-10);
    
    vertex = new SurfaceVertex(x2, y2, z2);
    p1 = projector.project(vertex.x, vertex.y, (vertex.z-zmin)*zfactor-10);
    
    BufferGC.drawLine(p0.x, p0.y, p1.x, p1.y);
    
  } 
 
  private final void DrawPoint(float x1, float y1, float z1) {
    
    SurfaceVertex vertex;
    Point p0;
    
    vertex = new SurfaceVertex(x1, y1, z1);
    p0 = projector.project(vertex.x, vertex.y, (vertex.z-zmin)*zfactor-10);
    
    BufferGC.drawLine(p0.x-3, p0.y, p0.x+3, p0.y);
    BufferGC.drawLine(p0.x, p0.y-3, p0.x, p0.y+3);
    
  } 
 
  /**
   * Determines whether a plane is plottable, i.e: does not have
   * invalid vertex.
   *
   * @return <code>true</code> if the plane is plottable, 
   *         <code>false</code> otherwise
   * @param values vertices array of the plane
   */

  private final boolean plottable(SurfaceVertex[] values) {
    return (!values[0].isInvalid() &&
            !values[1].isInvalid() &&
            !values[2].isInvalid() &&            
            !values[3].isInvalid()); 
  }

  private final SurfaceVertex values1[] = new SurfaceVertex[4];                  
  private final SurfaceVertex values2[] = new SurfaceVertex[4];                  
  
  /**
   * Plots an area of group of planes
   *
   * @param start_lx start index in x direction
   * @param start_ly start index in y direction
   * @param end_lx   end index in x direction
   * @param end_ly   end index in y direction
   * @param sx       step in x direction
   * @param sy       step in y direction
   */

  private final void plotArea(int start_lx, int start_ly,
                               int end_lx, int end_ly,
                               int sx, int sy) {
    start_lx *= calc_divisions + 1;
    sx *= calc_divisions + 1;
    end_lx *= calc_divisions + 1;

    int lx = start_lx;
    int ly = start_ly;

    while (ly != end_ly) {
      values1[1] = vertex[lx+ly];
      values1[2] = vertex[lx+ly+sy];
        
      while (lx != end_lx) {
        values1[0] = values1[1];
        values1[1] = vertex[lx+sx+ly];
        values1[3] = values1[2];
        values1[2] = vertex[lx+sx+ly+sy];

        if (plot_mode == DUALSHADE) color = 0.2f;
        if (plottable(values1)) plotPlane(values1,4);
        
        lx += sx; 
      }
      ly += sy; lx = start_lx;
    }
  }
 
  private final Point[] testpoint = new Point[5];
  
  /**
   * Creates a surface plot
   */

  private final void plotSurface() {
    float    zi, zx; //zMin,zMax
    int      sx, sy;
    int      start_lx, end_lx;
    int      start_ly, end_ly;
    
    image_drawn = false; 
    zi = Float.valueOf(frame.setting_panel.getZMin()).floatValue();
    zx = Float.valueOf(frame.setting_panel.getZMax()).floatValue();

    int plot_density = frame.setting_panel.getDispDivisions(); 
    int multiple_factor = calc_divisions / plot_density;
     
    zmin = zi; zmax = zx; 
    frame.function_panel.setMessage("erneuere ...");
    color_factor = 0.8f / (zmax-zmin);    
    if (plot_mode == DUALSHADE) color_factor *= 0.6f/0.8f;
    
    BufferGC.setColor(Color.lightGray);
    BufferGC.fillRect(0,0,bounds().width,bounds().height);
    drawBoxGridsTicksLabels(BufferGC,false);
    zfactor = 20/(zmax-zmin);
    
    // direction test

    float distance = projector.getDistance() * projector.getCosElevationAngle();

    // cop : center of projection
    
    cop = new SurfaceVertex((distance   * projector.getSinRotationAngle()) / projector.getXScaling(), 
                            (distance   * projector.getCosRotationAngle()) / projector.getYScaling(), 
                            (zmax-zmin) * ((projector.getDistance() * projector.getSinElevationAngle())
                            /projector.getZScaling()+10)/20 + zmin
                           );                     
    //cop.transform();
   
    boolean inc_x = cop.x > 0;
    boolean inc_y = cop.y > 0;
     
    if (inc_x) {
      start_lx = 0; end_lx = calc_divisions; 
      sx = multiple_factor;
    }
    else {
      start_lx = calc_divisions; end_lx = 0; 
      sx = -multiple_factor; 
    }     
    if (inc_y) {
      start_ly = 0; end_ly = calc_divisions; 
      sy = multiple_factor; 
    }
    else {
      start_ly = calc_divisions; end_ly = 0; 
      sy = -multiple_factor; 
    }

    if ((cop.x > 10) || (cop.x < -10)) {
      if ((cop.y > 10) || (cop.y < -10)) {
        plotArea(start_lx,start_ly,end_lx,end_ly,sx,sy);
      }
      else {    // split in y direction
        int split_y = (int)((cop.y+10)*plot_density/20)*multiple_factor;               
        plotArea(start_lx,0,end_lx,split_y,sx,multiple_factor);
        plotArea(start_lx,calc_divisions,end_lx,split_y,sx,-multiple_factor);        
      }
    }
    else {
      if ((cop.y > 10) || (cop.y < -10)) {   // split in x direction
        int split_x = (int)((cop.x+10)*plot_density/20)*multiple_factor;               
        plotArea(0,start_ly,split_x,end_ly,multiple_factor,sy);
        plotArea(calc_divisions,start_ly,split_x,end_ly,-multiple_factor,sy);        
      }
      else {    // split in both x and y directions
        int split_x = (int)((cop.x+10)*plot_density/20)*multiple_factor;               
        int split_y = (int)((cop.y+10)*plot_density/20)*multiple_factor;               
        plotArea(0,0,split_x,split_y,multiple_factor,multiple_factor);      
        plotArea(0,calc_divisions,split_x,split_y,multiple_factor,-multiple_factor);      
        plotArea(calc_divisions,0,split_x,split_y,-multiple_factor,multiple_factor);      
        plotArea(calc_divisions,calc_divisions,split_x,split_y,
                  -multiple_factor,-multiple_factor);      
      }
    }
    
    if (!(projector.project(-factor_x*10,-factor_y*10,-10).y < 
          projector.project(factor_x*10,factor_y*10,-10).y)) xyGrids(BufferGC);
    if (isBoxed) drawBoundingBox(); 
    frame.function_panel.setMessage("vollstndig"); 
    image_drawn = true; 
  }
  
}
