/*----------------------------------------------------------------------------------------* 
 * Parser.java version 1.0                                                    Jun 16 1996 *
 * Parser.java version 2.0                                                    Aug 25 1996 *
 * Parser.java version 2.1                                                    Oct 14 1996 *
 * Parser.java version 2.11                                                   Oct 25 1996 *
 * Parser.java version 2.2                                                    Nov  8 1996 *
 * Parser.java version 3.0                                                    May 17 1997 *
 * Parser.java version 3.01                                                   Oct 18 2001 *
 *                                                                                        *
 * Copyright (c) Yanto Suryono <yanto@fedu.uec.ac.jp>                                     *
 *                                                                                        *
 *                                                                                        *
 * MathOptAnimator   version 1.00     10 Feb 2002                                         *
 * Copyright (c) 2003 Ramazan Buzdemir <buzdemir@zpr.uni-koeln.de>                        *
 *                                                                                        *
 *                                                                                        *
 * This program is free software; you can redistribute it and/or modify it                *
 * under the terms of the GNU General Public License as published by the                  *
 * Free Software Foundation; either version 2 of the License, or (at your option)         *
 * any later version.                                                                     *
 *                                                                                        *
 * This program is distributed in the hope that it will be useful, but                    *
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or          *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for               *
 * more details.                                                                          *
 *                                                                                        *
 * You should have received a copy of the GNU General Public License along                *
 * with this program; if not, write to the Free Software Foundation, Inc.,                *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA                                  *
 *                                                                                        *
 *----------------------------------------------------------------------------------------*/

import java.util.Hashtable;
import java.util.Vector;

/**
 * Indicates that an error occured in parser operation, and the operation
 * could not be completed. Used internally in <code>Parser</code> class.
 *
 * @see Parser
 */

final class ParserException extends Exception {
  private int errorcode;
  
  /**
   * The constructor of <code>ParserException</code>.
   *
   * @param code the error code
   */
   
  ParserException(int code) {
    errorcode = code;
  }
  
  /**
   * Gets the error code.
   *
   * @return the error code
   */
   
  public int getErrorCode() {
    return errorcode;
  }
}

/**
 * The class <code>Parser</code> is a mathematical expression parser.<p>
 * Example of code that uses this class:<p>
 * 
 * <pre>
 * Parser parser = new Parser(1);    // creates parser with one variable
 * parser.defineVariable(1,"x");     // lets the variable be 'x'
 * parser.define("sin(x)/x");        // defines function: sin(x)/x
 * parser.parse();                   // parses the function
 * 
 * // calculates: sin(x)/x with x = -5.0 .. +5.0 in 20 steps
 * // and prints the result to standard output.
 *
 * float result;
 * for (i=-10; i <= 10; i++) {
 *   parser.setVariable(1,(float)i/2.0f);
 *   result = parser.evaluate();
 *   System.out.println(result);
 * }
 * </pre>  
 *
 * @author  Yanto Suryono
 */
 
public final class Parser {

  // global variables

  private int     var_count;             // number of variables
  private String  var_name[];            // variables' name   
  private double  var_value[];           // value of variables
  private double  number[];              // numeric constants in defined function
  private String  function = "";         // function definition
  private String  postfix_code = "";     // the postfix code
  private boolean valid = false;         // postfix code status      
  private int     error;                 // error code of last process

  // variables used during parsing
  
  private int     position;              // parsing pointer 
  private int     start;                 // starting position of identifier
  private int     num;                   // number of numeric constants 
  private char    character;             // current character

  // variables used during evaluating

  private boolean  radian;               // radian unit flag      
  private int      numberindex;          // pointer to numbers/constants bank

  private  static final int MAX_NUM       = 200;  // max numeric constants
  private  static final int NO_FUNCS      = 12;   // no. of built-in functions
  private  static final int STACK_SIZE    = 100;  // evaluation stack size

  // constants

  private  static final double DEGTORAD          = Math.PI / 180;
  private  static final double LOG10             = Math.log(10);
  
  // error codes

  /**
   * No error.
   */   
  public   static final int NO_ERROR             =  0;

  /**
   * Syntax error.
   */
  public   static final int SYNTAX_ERROR         =  1;

  /**
   * Parentheses expected.
   */
  public   static final int PAREN_EXPECTED       =  2;

  /**
   * Attempt to evaluate an uncompiled function.
   */
  public   static final int UNCOMPILED_FUNCTION  =  3;        

  /**
   * Expression expected.
   */
  public   static final int EXPRESSION_EXPECTED  =  4;

  /**
   * Unknown identifier.
   */
  public   static final int UNKNOWN_IDENTIFIER   =  5;

  /**
   * Operator expected.
   */
  public   static final int OPERATOR_EXPECTED    =  6;

  /**
   * Parenthesis mismatch.
   */
  public   static final int PAREN_NOT_MATCH      =  7;

  /**
   * Code damaged. 
   */
  public   static final int CODE_DAMAGED         =  8;

  /**
   * Stack overflow.
   */
  public   static final int STACK_OVERFLOW       =  9;

  /**
   * Too many constants.
   */
  public   static final int TOO_MANY_CONSTS      = 10;

  // postfix codes

  private  static final int  FUNC_OFFSET         = 1000;
  private  static final int  VAR_OFFSET          = 2000;
  private  static final char PI_CODE             = (char)253;
  private  static final char E_CODE              = (char)254;
  private  static final char NUMERIC             = (char)255;

  // built in functions 

  private String funcname[] = 
   
  { "sin",  "cos",  "tan",
    "asin", "acos", "atan",  
    "sinh", "cosh", "tanh",
    "ln",   "exp",  "sqrt" }; 

  /**
   * The constructor of <code>Parser</code>.
   *
   * @param variablecount the number of variables
   */
   
  Parser(int variablecount) {
    var_count = variablecount;
    radian = true;
    
    // arrays are much faster than vectors (IMHO)
    
    var_name = new String[variablecount]; 
    var_value = new double[variablecount];
    number = new double[MAX_NUM];     
  }

  /**
   * Sets the angle unit to radian. Default upon construction.
   */
   
  public void useRadian() {
    radian = true; 
  }

  /**
   * Sets the angle unit to degree. 
   */
    
  public void useDegree() {
    radian = false;
  }

  /**
   * Sets the variable names.
   * Nothing happens if variable index > number of variables.
   *
   * @param index the variable index (one based)
   * @param name  the variable name
   */
   
  public void defineVariable(int index, String name) {
    if (index > var_count) return; 
    var_name[index-1] = name;
  }

  /**
   * Sets the variable value.
   * The variable is accessed by index.
   * Nothing happens if variable index > number of variables.
   *
   * @param index the variable index (one based)
   * @param value the variable value
   */
   
  public void setVariable(int index, double value) {
    if (index > var_count) return;
    var_value[index-1] = value;
  }

  /**
   * Sets the variable value.
   * The variable is accessed by name.
   * Nothing happens if variable could not be found.
   *
   * @param name  the variable name
   * @param value the variable value
   */

  public void setVariable(String name, double value) {
    for (int i=0; i < var_count; i++) {
      if (var_name[i].equals(name)) {
        var_value[i] = value; break;
      }
    }
  }

  /**
   * Defines a function. Current postfix code becomes invalid.
   *
   * @param definition the function definition
   */

  public void define(String definition) {
    function = definition.toLowerCase();
    valid = false;
  }
 
  /**
   * Parses defined function.
   */
   
  public void parse() {
    String allFunction = new String(function);
    String orgFunction = new String(function);
    int index;

    if (valid) return;     
    num = 0; error = NO_ERROR;    

    if (error == NO_ERROR) {
      function = allFunction + ')';
      parseSubFunction();
    }
    function = orgFunction;        
    valid = (error == NO_ERROR);
  }
  
  /**
   * Evaluates compiled function.
   *
   * @return the result of the function
   */ 

  public double evaluate() {
    double result;
    
    if (!valid) {
      error = UNCOMPILED_FUNCTION; return 0;
    }
    error = NO_ERROR; numberindex = 0;
    result = evaluateSubFunction();
    return result;
  }  

  /**
   * Gets error code of last operation.
   *
   * @return the error code
   */
   
  public int getErrorCode() {
    return error;
  }
  
  /**
   * Gets error string/message of last operation.
   *
   * @return the error string 
   */

  public String getErrorString() {
    return toErrorString(error);
  }

  /**
   * Gets error position. Valid only if error code != NO_ERROR
   *
   * @return error position (one based)
   */
   
  public int getErrorPosition() {
    return position;
  }
  
  /**
   * Converts error code to error string.
   *
   * @return the error string
   */
    
  public String toErrorString(int errorcode) {
    String s = "";
   
    switch (errorcode) {
      case NO_ERROR             : s = "Kein Fehler"; break;
      case SYNTAX_ERROR         : s = "Syntax Fehler"; break;
      case PAREN_EXPECTED       : s = "Klammer erwartet"; break;
      case UNCOMPILED_FUNCTION  : s = "Unkompilierte Funktion"; break;
      case EXPRESSION_EXPECTED  : s = "Ausdruck erwartet"; break;
      case UNKNOWN_IDENTIFIER   : s = "Unbekannter Bezeichner"; break;
      case OPERATOR_EXPECTED    : s = "Operator erwartet"; break;
      case PAREN_NOT_MATCH      : s = "Klammer erwartet"; break;
      case CODE_DAMAGED         : s = "Interner Code beschdigt"; break;
      case STACK_OVERFLOW       : s = "Stack berlauf"; break; 
      case TOO_MANY_CONSTS      : s = "Zu viele Konstanten"; break;
    }
    return s;
  }  

/*----------------------------------------------------------------------------------------*
 *                            Private methods begin here                                  *
 *----------------------------------------------------------------------------------------*/

  /**
   * Advances parsing pointer, skips pass all white spaces.
   *
   * @exception ParserException 
   */
   
  private void skipSpaces() throws ParserException {
    try {
      while (function.charAt(position-1) == ' ') position++;
      character = function.charAt(position-1); 
    }
    catch(StringIndexOutOfBoundsException e) {
      throw new ParserException(PAREN_NOT_MATCH);
    }
  }

  /**
   * Advances parsing pointer, gets next character.
   *
   * @exception ParserException
   */
   
  private void getNextCharacter() throws ParserException {
    position++;
    try {
      character = function.charAt(position-1);
    }
    catch(StringIndexOutOfBoundsException e) {
      throw new ParserException(PAREN_NOT_MATCH);
    }
  }
 
  /**
   * Appends postfix code to compiled code.
   *
   * @param code the postfix code to append
   */
   
  private void addCode(char code) {
    postfix_code += code;
  }

  /**
   * Scans a number. Valid format: xxx[.xxx[e[+|-]xxx]]
   *
   * @exception ParserException
   */
   
  private void scanNumber() throws ParserException {
    String  numstr = "";
    double  value;

    if (num == MAX_NUM) 
      throw new ParserException(TOO_MANY_CONSTS);
    do {
      numstr += character; getNextCharacter(); 
    } while ((character >= '0') && (character <= '9'));
    if (character == '.') {
      do {
        numstr += character; getNextCharacter();
      } while ((character >= '0') && (character <= '9'));
    }
    if (character == 'e') {
      numstr += character; getNextCharacter();
      if ((character == '+') || (character == '-')) {
        numstr += character; getNextCharacter();
      }
      while ((character >= '0') && (character <= '9')) {
        numstr += character; getNextCharacter();
      }
    }
    try {
      value = Double.valueOf(numstr).doubleValue();
    }
    catch(NumberFormatException e) {
      position = start;
      throw new ParserException(SYNTAX_ERROR); 
    }
    number[num++] = value; addCode(NUMERIC);
  }

  /**
   * Scans a non-numerical identifier. Can be function call,
   * variable, etc.
   *
   * @exception ParserException
   */
   
  private void scanNonNumeric() throws ParserException {
    String stream = "";

    if ((character == '*') || (character == '/') ||
        (character == '^') || (character == ')'))    
      throw new ParserException(SYNTAX_ERROR);
    do {
      stream += character; getNextCharacter();
    } while (!(
              (character == ' ') || (character == '+') || 
              (character == '-') || (character == '*') ||
              (character == '/') || (character == '^') || 
              (character == '(') || (character == ')')
              )); 
    if (stream.equals("pi")) {
      addCode(PI_CODE); return;
    }
    else
    if (stream.equals("e")) {
      addCode(E_CODE); return;
    }
 
    // built-in function

    for (int i = 0; i < NO_FUNCS; i++) {
      if (stream.equals(funcname[i])) {
        skipSpaces(); 
        if (character != '(') 
          throw new ParserException(PAREN_EXPECTED); 
        scanAndParse();
        if (character != ')') 
          throw new ParserException(PAREN_EXPECTED);    
        getNextCharacter();
        addCode((char)(i + FUNC_OFFSET));       
        return; 
      }
    }

    // registered variables
    
    for (int i = 0; i < var_count; i++) {
      if (stream.equals(var_name[i])) {
        addCode((char)(i + VAR_OFFSET)); return;  
      }
    }
       
    position = start;
    throw new ParserException(UNKNOWN_IDENTIFIER);  
  }
  
  /**
   * Gets an identifier starting from current parsing pointer.
   *
   * @return whether the identifier should be negated
   * @exception ParserException
   */
   
  private boolean getIdentifier() throws ParserException {
    boolean negate = false;

    getNextCharacter(); skipSpaces(); 
    
    while ((character == '+') || (character == '-')) {
      if (character == '-') negate = !negate; 
      getNextCharacter(); skipSpaces(); 
    }
    
    start = position;
    if ((character >= '0') && (character <= '9'))
      scanNumber();
    else
    if (character == '(') {
      scanAndParse();
      getNextCharacter();
    }
    else scanNonNumeric();
    skipSpaces(); 
    return (negate);
  }

  /**
   * Scans arithmetic level 3 (highest). Power arithmetics.
   *
   * @exception ParserException
   */
   
  private void arithmeticLevel3() throws ParserException {
    boolean negate;
//System.out.println("level 3");
    negate = getIdentifier();
    if (negate) addCode('_');
    addCode('^');
    if (character == '^') arithmeticLevel3();
    if (character == '*') arithmeticLevel2();
  }

  /**
   * Scans arithmetic level 2. Multiplications and divisions.
   *
   * @exception ParserException
   */

  private void arithmeticLevel2() throws ParserException {
    boolean negate;
//System.out.println("level 2");  	
    do {
      char operator = character;
      negate = getIdentifier();
      if (character == '^') arithmeticLevel3();
      if (negate) addCode('_');
      addCode(operator);
    } while ((character == '*') || (character == '/'));
  }

  /**
   * Scans arithmetic level 1 (lowest). 
   * Additions and substractions.
   *
   * @exception ParserException
   */

  private void arithmeticLevel1() throws ParserException {
    boolean negate;
//System.out.println("level 1");  	
    do {
      char operator = character;
      negate = getIdentifier();   
      if (character == '^') {
      	arithmeticLevel3();
        if (negate) addCode('_');
      }
      else
      if ((character == '*') || (character == '/')) {
      	if (negate) addCode('_');
      	arithmeticLevel2();
      }
      else
      if (negate) addCode('_'); 
      addCode(operator);           
    } while ((character == '+') || (character == '-'));
  }

  /**
   * Main method of scanning and parsing process.
   *
   * @exception ParserException
   */

  private void scanAndParse() throws ParserException {
    boolean negate;
    
    negate = getIdentifier();

    if ((character != '^') && (negate)) addCode('_');
    
    do {
//System.out.println("operator: " + character);
      switch (character) {
        case '+':
        case '-': arithmeticLevel1(); break;
        case '*': 
        case '/': arithmeticLevel2(); break;
        case '^': arithmeticLevel3(); 
        	  if (negate) addCode('_');
        	  break;
        case ')': return;

        default : throw new ParserException(OPERATOR_EXPECTED);
      }
    } while (true);
  }

  /**
   * Parses subfunction.
   */
   
  private void parseSubFunction() {
    position = 0; 
    postfix_code = "";
    try {
      scanAndParse();
    }
    catch (ParserException e) {
      error = e.getErrorCode();
      if ((error == SYNTAX_ERROR) && (postfix_code == "")) 
      error = EXPRESSION_EXPECTED;
    }
    if ((error == NO_ERROR) && 
        (position != function.length())) error = PAREN_NOT_MATCH;
  }
  
  /**
   * Built-in one parameter function call.
   *
   * @return the function result
   * @param  function  the function index
   * @param  parameter the parameter to the function
   */
   
  private double builtInFunction(int function, double parameter) {
    switch (function) {
      case  0: if (radian)
                 return Math.sin(parameter);
               else
                 return Math.sin(parameter * DEGTORAD);
      case  1: if (radian)
                 return Math.cos(parameter);
               else
                 return Math.cos(parameter * DEGTORAD);
      case  2: if (radian)
                 return Math.tan(parameter);
               else
                 return Math.tan(parameter * DEGTORAD);
      case  3: if (radian)
                 return Math.asin(parameter);
               else
                 return Math.asin(parameter) / DEGTORAD;
      case  4: if (radian)
                 return Math.acos(parameter); 
               else
                 return Math.acos(parameter) / DEGTORAD;
      case  5: if (radian)
                 return Math.atan(parameter);
               else
                 return Math.atan(parameter) / DEGTORAD;                 
      case  6: return (Math.exp(parameter)-Math.exp(-parameter))/2;
      case  7: return (Math.exp(parameter)+Math.exp(-parameter))/2;
      case  8: double a = Math.exp(parameter);
               double b = Math.exp(-parameter);
               return (a-b)/(a+b);
      case  9: return Math.log(parameter);             
      case 10: return Math.exp(parameter);
      case 11: return Math.sqrt(parameter);
      default: error = CODE_DAMAGED; 
               return Double.NaN;
    }
  }
   
  /**
   * Evaluates subfunction.
   *
   * @return the result of the subfunction
   */
   
  private double evaluateSubFunction() {
    double stack[];
    int    stack_pointer = -1;
    int    code_pointer = 0;
    int    destination;
    char   code;
    
//System.out.println("postfix: " + postfix_code);
    stack = new double[STACK_SIZE]; 
    while (true) {
      try {
        code = postfix_code.charAt(code_pointer++);
      }
      catch (StringIndexOutOfBoundsException e) {
        return stack[0];       
      }
      try {
        switch (code) {
          case '+': 
//System.out.println(stack[stack_pointer-1] + " + " + stack[stack_pointer]);                    
                    stack[stack_pointer-1] += stack[stack_pointer];
                    stack_pointer--; break;
          case '-': 
//System.out.println(stack[stack_pointer-1] + " - " + stack[stack_pointer]);                    
                    stack[stack_pointer-1] -= stack[stack_pointer];          
                    stack_pointer--; break; 
          case '*': 
//System.out.println(stack[stack_pointer-1] + " * " + stack[stack_pointer]);                    
                    stack[stack_pointer-1] *= stack[stack_pointer];
                    stack_pointer--; break; 
          case '/': 
//System.out.println(stack[stack_pointer-1] + " / " + stack[stack_pointer]);                    
                    stack[stack_pointer-1] /= stack[stack_pointer]; 
                    stack_pointer--; break;
          case '^': 
//System.out.println(stack[stack_pointer-1] + " ^ " + stack[stack_pointer]);                    
                    stack[stack_pointer-1] = Math.pow(stack[stack_pointer-1], 
                                                      stack[stack_pointer]);
                    stack_pointer--; break;
          case '_': 
//System.out.println("-" + stack[stack_pointer]);          
                    stack[stack_pointer] = -stack[stack_pointer]; break;
 
          case NUMERIC       : stack[++stack_pointer] = number[numberindex++]; break;
          case PI_CODE       : stack[++stack_pointer] = Math.PI; break;
          case E_CODE        : stack[++stack_pointer] = Math.E; break;
                
          default: if ((int)code >= VAR_OFFSET) 
                     stack[++stack_pointer] = var_value[(int)code-VAR_OFFSET];
                   else
                   if ((int)code >= FUNC_OFFSET) {
                     stack[stack_pointer] = 
                     builtInFunction((int)code-FUNC_OFFSET, stack[stack_pointer]);
                   }
                   else {
                     error = CODE_DAMAGED;
                     return Double.NaN;
                   }
        }
      }
      catch (ArrayIndexOutOfBoundsException oe) {
        error = STACK_OVERFLOW;       
        return Double.NaN;
      }
      catch (NullPointerException ne) {
        error = CODE_DAMAGED;
        return Double.NaN;
      }
    }
  }
}

