/*----------------------------------------------------------------------------------------*
 * Algorithm.java                                                                         *
 *                                                                                        *
 *                                                                                        *
 * MathOptAnimator   version 1.00     10 Feb 2002                                         *
 * Copyright (c) 2003 Ramazan Buzdemir <buzdemir@zpr.uni-koeln.de>                        *
 *                                                                                        *
 *                                                                                        *
 * This program is free software; you can redistribute it and/or modify it                *
 * under the terms of the GNU General Public License as published by the                  *
 * Free Software Foundation; either version 2 of the License, or (at your option)         *
 * any later version.                                                                     *
 *                                                                                        *
 * This program is distributed in the hope that it will be useful, but                    *
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or          *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for               *
 * more details.                                                                          *
 *                                                                                        *
 * You should have received a copy of the GNU General Public License along                *
 * with this program; if not, write to the Free Software Foundation, Inc.,                *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA                                  *
 *                                                                                        *
 *----------------------------------------------------------------------------------------*/

import com.javathings.math.Derive;

public class Algorithm {
  private SurfaceFrame frame;
  
  private String functiontype;
  private String algorithmtype;
  private double epsilon;
  private int kmax;
  
  public int k;
  private Derive d;
  private boolean trace, error, fertig;
  
  private String string_f;
  private String string_dx;
  private String string_dy;
  private String string_dxdx;   // q11
  private String string_dydx;   // q12
  private String string_dxdy;   // q21
  private String string_dydy;   // q22
  
  private Parser parser_f;
  private Parser parser_dx;
  private Parser parser_dy;
  private Parser parser_dxdx;
  private Parser parser_dydx;
  private Parser parser_dxdy;
  private Parser parser_dydy;
    
  private double xk, yk;                 // (Vektor xk_) =(xk,yk)
  private double mk, nk;                 // (Gradient gradk) = (mk,nk)
  private double mk_1, nk_1;
  private double ak, bk;                 // (Abstiegsrichtung dk) = (ak,bk)
  private double normk;
  private double zk;  
  private double tk, alphak;
  private double q11, q12, q21, q22;     // Hessematrix
  private double zaehler, nenner;
  private double h0, hs0;
  
  private final double p1   = 1.0e-04;   // Wolfe-Bedingung 1
  private final double p2   = 1.0e-01;   // Wolfe-Bedingung 2
  
  private double min_x, max_x;
  private double min_y, max_y;
  private double min_z, max_z;
  private double xs, ys, zs;
  
  public float points_x[];
  public float points_y[];
  
  Algorithm(SurfaceFrame frame) {
    this.frame = frame;

    parser_dx = new Parser(2);
    parser_dx.defineVariable(1,"x");
    parser_dx.defineVariable(2,"y");
    
    parser_dy = new Parser(2);
    parser_dy.defineVariable(1,"x");
    parser_dy.defineVariable(2,"y");
    
    parser_dxdx = new Parser(2);
    parser_dxdx.defineVariable(1,"x");
    parser_dxdx.defineVariable(2,"y");
    
    parser_dydx = new Parser(2);
    parser_dydx.defineVariable(1,"x");
    parser_dydx.defineVariable(2,"y");
    
    parser_dxdy = new Parser(2);
    parser_dxdy.defineVariable(1,"x");
    parser_dxdy.defineVariable(2,"y");
    
    parser_dydy = new Parser(2);
    parser_dydy.defineVariable(1,"x");
    parser_dydy.defineVariable(2,"y");
      
    d = new Derive();
  }

  public void start() {

//-------------------------------------------------------------------------------------------
    
    trace = true;
    frame.algorithm_panel.output.setText("");
    frame.algorithm_panel.output.append
      ("==========================================================================");
    frame.algorithm_panel.output.append
      ("\n  " + "Ausgabe:" + "\n");
    frame.algorithm_panel.output.append
      ("==========================================================================");


    functiontype = frame.algorithm_panel.getFunctionType();
    if (functiontype == "quadratisch") {
      algorithmtype = "Fletcher-Reeves";
      epsilon = 0.0;
      kmax = 2;
    }
    else {
      algorithmtype = frame.algorithm_panel.getAlgorithmType();
      epsilon = Math.pow((double)10,
                         (double)Integer.parseInt(frame.algorithm_panel.getBreakCondition()));
      kmax = Integer.parseInt(frame.algorithm_panel.getMaxIteration());
    }
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    string_f = frame.function_panel.getFunctionDefinition();
    try{
      String gradient[] = d.diff(string_f, "x;y");
      String hesse0[]   = d.diff(gradient[0], "x;y");
      String hesse1[]   = d.diff(gradient[1], "x;y");
      string_dx   = gradient[0];
      string_dy   = gradient[1];
      string_dxdx = hesse0[0];
      string_dydx = hesse0[1];
      string_dxdy = hesse1[0];
      string_dydy = hesse1[1];
    }
    catch(Exception e){
      frame.algorithm_panel.output.append
        ("\n  " + "Fehler:" +
         "\n  " + "Der Gradient oder die Hessematrix von f konnte nicht gebildet werden." +
         "\n  " + "Bitte geben Sie eine andere Funktion ein und starten Sie den Algorithmus neu.");
      stop();
      return;
    }
    
//System.out.println("--------------------------------------------------------------");
//System.out.println("f " + string_f);
//System.out.println("dx " + string_dx);
//System.out.println("dy " + string_dy);
//System.out.println("dxdx " + string_dxdx);
//System.out.println("dydx " + string_dydx);
//System.out.println("dxdy " + string_dxdy);
//System.out.println("dydy " + string_dydy);
//System.out.println("--------------------------------------------------------------");
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    error = false;
    
    parser_f = frame.parser;

    parser_dx.define(string_dx);
    parser_dx.parse();
    if (parser_dx.getErrorCode() != parser_dx.NO_ERROR) error = true;
    
    parser_dy.define(string_dy);
    parser_dy.parse();
    if (parser_dy.getErrorCode() != parser_dy.NO_ERROR) error = true;
    
    if (error) {
      frame.algorithm_panel.output.append
        ("\n  " + "Fehler:" +
         "\n  " + "Der Gradient von f konnte nicht gebildet werden." +
         "\n  " + "Bitte geben Sie eine andere Funktion ein und starten Sie den Algorithmus neu.");
      stop();
      return;
    }

    parser_dxdx.define(string_dxdx);
    parser_dxdx.parse();
    if (parser_dxdx.getErrorCode() != parser_dxdx.NO_ERROR) error = true;

    parser_dydx.define(string_dydx);
    parser_dydx.parse();
    if (parser_dydx.getErrorCode() != parser_dydx.NO_ERROR) error = true;
    
    parser_dxdy.define(string_dxdy);
    parser_dxdy.parse();
    if (parser_dxdy.getErrorCode() != parser_dxdy.NO_ERROR) error = true;

    parser_dydy.define(string_dydy);
    parser_dydy.parse();
    if (parser_dydy.getErrorCode() != parser_dydy.NO_ERROR) error = true;

    if (error) {
      frame.algorithm_panel.output.append
        ("\n  " + "Fehler:" +
         "\n  " + "Die Hessematrix von f konnte nicht gebildet werden." +
         "\n  " + "Bitte geben Sie eine andere Funktion ein und starten Sie den Algorithmus neu.");
      stop();
      return;
    }
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    // Test, ob eingegebene Funktion quadratisch.
    
    if (functiontype == "quadratisch") {
      error = false;
      isConstant(string_dxdx);
      isConstant(string_dydx);
      isConstant(string_dxdy);
      isConstant(string_dydy);
      if (error) {
    	frame.algorithm_panel.output.append
      	  ("\n  " + "Fehler:" +
      	   "\n  " + "Die eingegebene Funktion ist nicht quadratisch." +
           "\n  " + "Bitte geben Sie eine quadratische Funktion ein und starten Sie den Algorithmus neu.");
        stop();
        return;
      }
      
      q11 = eval(parser_dxdx,0.0,0.0); if (error) return;
      q12 = eval(parser_dydx,0.0,0.0); if (error) return;
      q21 = eval(parser_dxdy,0.0,0.0); if (error) return;
      q22 = eval(parser_dydy,0.0,0.0); if (error) return;
//System.out.println(q11);
//System.out.println(q12);
//System.out.println(q21);
//System.out.println(q22);

      if (q12 != q21) {
        frame.algorithm_panel.output.append
          ("\n  " + "Fehler:" +
      	   "\n  " + "Die eingegebene Funktion ist nicht quadratisch." +
           "\n  " + "Bitte geben Sie eine quadratische Funktion ein und starten Sie den Algorithmus neu.");
        stop();
        return;
      }
      
      if (!( (q11 > 0.0) && (q11*q22-q12*q21 > 0.0) )) {
        frame.algorithm_panel.output.append
          ("\n  " + "Die eingegebene quadratische Funktion besitzt kein strenges Minimum," +
           "\n  " + "da ihre Hessematrix nicht positiv definit ist.");
        stop();
        return;
      }
                  
    }
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    k = 0;
//System.out.println("k = " + k);

    frame.algorithm_panel.output.append
      ("\n  " + "k = " + k);
//-------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------- 
    // (Startvektor xk_) = (xk,yk)
    xk = Double.valueOf(frame.algorithm_panel.getX0()).doubleValue();
    yk = Double.valueOf(frame.algorithm_panel.getY0()).doubleValue();
//System.out.println("(xk,yk) = " + xk + "," + yk);

    frame.algorithm_panel.output.append
      ("\n  " + "Vektor (xk,yk) = ( " + xk + " , " + yk  + " )");
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    // zk = f(xk,yk)
    zk = eval(parser_f, xk, yk); if (error) return;
//System.out.println("f(xk,yk) = " + zk);

    frame.algorithm_panel.output.append
      ("\n  " + "Funktionswert f(xk,yk) = " + zk);
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    // (Gradient gradk) = (mk,nk)
    mk = eval(parser_dx, xk, yk); if (error) return;
    nk = eval(parser_dy, xk, yk); if (error) return;
//System.out.println("(mk,nk) = " + mk + "," + nk);

    frame.algorithm_panel.output.append
      ("\n  " + "Gradient grad(xk,yk) = ( " + mk + ", " + nk  + " )");
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    // (Norm ||grad(xk,yk)||) = sqrt(mk^2+nk^2)
    normk = Math.sqrt( (double)(mk*mk)+(double)(nk*nk) );
//System.out.println("||grad(xk,yk)|| = " + normk);
//System.out.println("--------------------------------------------------------------");
    
    frame.algorithm_panel.output.append
      ("\n  " + "Norm ||grad(xk,yk)|| = " + normk);
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    // (Abstiegsrichtung dk) = (ak,bk)
    ak = -mk;
    bk = -nk;
//System.out.println("(ak,bk) = "+ ak + "," + bk);

    if (normk > epsilon)
      frame.algorithm_panel.output.append
        ("\n  " + "Abstiegsrichtung dk = ( " + ak + ", " + bk  + " )");
        
    frame.algorithm_panel.output.append
      ("\n" + "==========================================================================");      
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
    // Radius um (x*,y*) mindestens 2

    int _k = k;
    double _xk = xk;
    double _yk = yk;
    double _mk = mk;
    double _nk = nk;
    double _ak = ak;
    double _bk = bk;
    double _normk = normk;
    double _zk = zk;
    double _epsilon = epsilon;
    int _kmax = kmax;
    
    min_x = xk; max_x = xk;
    min_y = yk; max_y = yk;
    min_z = zk; max_z = zk;
    epsilon = 1.0e-3;

    points_x = null;
    points_y = null;
    points_x = new float[kmax+1];
    points_y = new float[kmax+1];
  
    trace = false;
    fertig = false;
    while (!fertig) {
      if (xk < min_x) min_x = xk;
      if (xk > max_x) max_x = xk;
      if (yk < min_x) min_y = yk;
      if (yk > max_x) max_y = yk;
      if (zk < min_z) min_z = zk;
      if (zk > max_z) max_z = zk;
      next();
    }
    fertig = false;
    trace = true;
    
//System.out.println("min_x: " + min_x);
//System.out.println("min_y: " + min_y);
//System.out.println("min_z: " + min_z);
//System.out.println("max_x: " + max_x);
//System.out.println("max_y: " + max_y);
//System.out.println("max_z: " + max_z);
//System.out.println("********************"); 
    
    xs = xk; ys = yk; zs = zk;
  
    max_x = Math.max(Math.abs(min_x-xs),Math.abs(max_x-xs));
    max_y = Math.max(Math.abs(min_y-ys),Math.abs(max_y-ys));
    if (max_x == 0.0)
      max_x = 3.0;
    else
      max_x = Math.max(max_x,max_y) * 1.1;
  
    max_z = Math.max(Math.abs(min_z-zs),Math.abs(max_z-zs));
    if (max_z == 0.0)
      max_z = 2.0;
    else {
      min_z = Math.min(Math.abs(eval(parser_f,xs-max_x,ys-max_x)-zs),
                       Math.abs(eval(parser_f,xs-max_x,ys+max_x)-zs)); if (error) return;
      min_z = Math.min(min_z, Math.abs(eval(parser_f,xs+max_x,ys-max_x)-zs)); if (error) return;
      min_z = Math.min(min_z, Math.abs(eval(parser_f,xs+max_x,ys+max_x)-zs)); if (error) return;
      max_z = Math.max(max_z*1.1,min_z);
    }    
//System.out.println("max_x: " + max_x);
//System.out.println("max_y: " + max_y);
//System.out.println("max_z: " + max_z);
//System.out.println("********************");

    k  = _k;
    xk = _xk;
    yk = _yk;
    mk = _mk;
    nk = _nk;
    ak = _ak;
    bk = _bk;
    normk = _normk;
    zk = _zk;
    epsilon = _epsilon;
    kmax = _kmax;
    
    points_x[k] = (float)xk;
    points_y[k] = (float)yk;

    frame.setting_panel.setRanges(String.valueOf(xs-max_x), String.valueOf(xs+max_x),
                                  String.valueOf(ys-max_x), String.valueOf(ys+max_x),
                                  String.valueOf(zs-max_z), String.valueOf(zs+max_z));
    
    if (frame.plot(false,false) == false)
      frame.setting_panel.setRanges("-3.0","3.0","-3.0","3.0","-2.0","2.0");
      
    frame.plot(false,true);

    frame.algorithm_panel.setEnabled(false);
//-------------------------------------------------------------------------------------------
                                        
  }
  
  public void stop() {
    frame.algorithm_panel.setEnabled(true);
  }
  
  public void next() {

    if ( (k < kmax) && (normk > epsilon) ) {
    	
//------------------------------------------------------------------------------------------- 
      // Inkrementiere k	
      k++;
//System.out.println("k = " + k);

      if (trace) {
        frame.algorithm_panel.output.append
          ("\n  " + "k = " + k);
      }
//------------------------------------------------------------------------------------------- 

//-------------------------------------------------------------------------------------------
      // Berechne Minimum tk von h(t)=f((xk+t*ak),(yk+t*bk))
      
      if (functiontype == "quadratisch") {
      	// h'(t) = grad(xk_ + t*dk)             * (dk)T
      	//       = ((xk_ + t*dk)*hessek-p))     * (dk)T
      	//       = (xk_*hessek-p + t*dk*hessek) * (dk)T
      	//       = (grad(xk_) + t*dk*hessek)    * (dk)T
      	//       = gradk*(dk)T + t*dk*hessek*(dk)T
      	// h'(t) = 0
      	//   ==> tk = -(gradk*(dk)T) / (dk*hessek*(dk)T)
      	//          = -((mk,nk)*(ak,bk)T) / ((ak,bk)*(q11,q12,q21,q22)*(ak,bk)T)
      	//          = -(mk*ak+nk*bk) / (q11*ak + q21*bk, q12*ak + q22*bk)*(ak,bk)T)
      	//          = -(mk*ak+nk*bk) / (q11*ak*ak + q21*ak*bk + q12*ak*bk + q22*bk*bk)
      	nenner  = q11*ak*ak + q21*ak*bk + q12*ak*bk + q22*bk*bk;
      	zaehler = mk*ak + nk*bk;
      	tk = - zaehler / nenner;
      }
      else {
        
        // h0 = h(0) = f(xk_) = f(xk,yk) = zk
        // hs0 = h'(0) = grad(xk_)T * dk = (mk,nk)T * (ak,bk) = mk*ak + nk*bk  
        h0 = zk;
        hs0 = mk*ak + nk*bk;
        
        if (hs0 >= 0) {
      	  if (trace) {
      	    frame.algorithm_panel.output.append
      	      ("\n  " + "Der Richtungsvektor dk ist keine Abstiegsrichtung." +
               "\n  " + "D.h. in dieser Richtung kann kein Minimum (x*,y*) gefunden werden.");
            stop();
            k--;
          }
          fertig = true;
          return;
        }        	
        
        tk = Newton();
        if (tk == 0.0)
          tk = LineSearch();
        
        if (error) return;
        
        if (tk == 0.0) {
          if (trace) {
            frame.algorithm_panel.output.append
              ("\n  " + "Es konnte keine gltige Schrittweite tk > 0 bestimmt werden." +
               "\n  " + "Wahrscheinlich ist die eingegebene Funktion in Richtung von dk nicht nach unten" +
               "\n  " + "beschrnkt oder aber das Verfahren zur Berechnung von tk hat hier versagt." +
               "\n  " + "Bitte geben Sie entweder einen anderen Startvektor (x0,y0) oder eine andere Funktion ein," + 
               "\n  " + "und starten Sie den Algorithmus neu.");
            stop();
            k--;
          }
          fertig = true;
          return;
        }
               
      } //else
//System.out.println("tk = " + tk);


      if (trace) {
        frame.algorithm_panel.output.append("\n  " + "Schrittweite tk = " + tk);
      }
//-------------------------------------------------------------------------------------------
  
//-------------------------------------------------------------------------------------------      
      // Berechne xk=xk+tk*ak
      // Berechne yk=yk+tk*bk
      xk = xk + tk*ak;
      yk = yk + tk*bk;
//System.out.println("(xk,yk) = " + xk + "," + yk);
      
      if (trace) {
        frame.algorithm_panel.output.append
          ("\n  " + "Vektor (xk,yk) = ( " + xk + " , " + yk  + " )");
      }
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
      // zk = f(xk,yk)
      zk = eval(parser_f, xk, yk); if (error) return;
//System.out.println("f(xk,yk) = " + zk);
    
      if (trace) {
        frame.algorithm_panel.output.append
          ("\n  " + "Funktionswert f(xk,yk) = " + zk);
      }
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
      // Berechne mk=eval(dx,xk,yk)
      // Berechne nk=eval(dy,xk,yk)
      
      mk_1 = mk;
      nk_1 = nk;
      
      mk = eval(parser_dx, xk, yk); if (error) return;
      nk = eval(parser_dy, xk, yk); if (error) return;
//System.out.println("(mk,nk) = " + mk + "," + nk);

      if (trace) {
        frame.algorithm_panel.output.append
          ("\n  " + "Gradient grad(xk,yk) = ( " + mk + ", " + nk  + " )");
      }
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------       
      // (Norm ||grad(xk,yk)||) = sqrt(mk^2+nk^2)
      normk = Math.sqrt( (double)(mk*mk)+(double)(nk*nk) );
//System.out.println("||grad(xk,yk)|| = " + normk);
//System.out.println("--------------------------------------------------------------");
    
      if (trace)
        frame.algorithm_panel.output.append
          ("\n  " + "Norm ||grad(xk,yk)|| = " + normk);
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------     
      if (normk > epsilon) {
        // Berechne alphak
        if (algorithmtype == "Hestenes-Stiefel")
          nenner = ak*(mk-mk_1) + bk*(nk-nk_1);
        if (algorithmtype == "Fletcher-Reeves")
          nenner = mk_1*mk_1 + nk_1*nk_1;
        if (algorithmtype == "Polak-Ribiere")
          nenner = mk_1*mk_1 + nk_1*nk_1;      
//System.out.println("nenner alphak = " + nenner);

        // unwahrscheinlich, zur Sicherheit
        if (nenner == 0.0) {
      	  if (trace) {
      	    frame.algorithm_panel.output.append
      	      ("\n  " + "alphak konnte nicht berechnet werden." +
               "\n  " + "Die eingegebene Funktion besitzt wahrscheinlich keinen kritischen Punkt.");
            stop();
            k--;
          }
          fertig = true;
      	  return;
        }

        if (algorithmtype == "Hestenes-Stiefel")
          zaehler = mk*(mk-mk_1) + nk*(nk-nk_1);
        if (algorithmtype == "Fletcher-Reeves")
          zaehler = mk*mk + nk*nk; 
        if (algorithmtype == "Polak-Ribiere")
          zaehler = mk*(mk-mk_1) + nk*(nk-nk_1);
             
        alphak = zaehler / nenner;
//System.out.println("alphak = " + alphak);
//-------------------------------------------------------------------------------------------
 
//-------------------------------------------------------------------------------------------     
        // Berechne ak=-mk+alphak*ak
        // Berechne bk=-nk+alphak*bk
        ak = -mk + alphak*ak;
        bk = -nk + alphak*bk;
//System.out.println("(ak,bk) = " + ak + "," + bk);      
      
        if (trace)
          frame.algorithm_panel.output.append
            ("\n  " + "Abstiegsrichtung dk = ( " + ak + ", " + bk  + " )");
      }
      
      if (trace)      
        frame.algorithm_panel.output.append
          ("\n" + "==========================================================================");
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
      if (trace) {
        
        points_x[k] = (float)xk;
        points_y[k] = (float)yk;
    
        frame.plot(false,true);
      }
//-------------------------------------------------------------------------------------------

    }
    else {
    
      if (trace) {	

        if (functiontype == "quadratisch") {
          if (normk > 0.01) {
            frame.algorithm_panel.output.append
              ("\n  " + "Der gefundene Vektor (x2,y2) ist nicht das gesuchte strenge Minimum der eingegebenen" +
               "\n  " + "quadratischen Funktion, da ||gradf(x2,y2)|| viel grsser als 0 ist." +
               "\n  " + "Das Minimum konnte wahrscheinlich aufgrund interner Rundungsfehler nicht nach zwei" + 
               "\n  " + "Iterationsschritten gefunden werden. Der Startvektor (x0,y0) sollte mglichst nahe am" +
               "\n  " + "Minimum liegen und nicht aus zu grossen Zahlen x0 und y0 bestehen." +
               "\n  " + "Geben Sie entweder einen neuen Startvektor (x0,y0) ein oder starten Sie den Algorithmus" +
               "\n  " + "fr differenzierbare Funktionen. Im zweiten Fall knnen so mehr als nur zwei" +
               "\n  " + "Iterationschritte ausgefhrt werden, um das gesuchte strenge Minimum zu finden.");
          }
          else {
            frame.algorithm_panel.output.append
              ("\n  " + "Das gesuchte strenge Minimum der eingegebenen quadratischen Funktion ist  " +
               "\n  " + "(x*,y*) = ( " + xk + " , " + yk + " )" +
               "\n  " + "mit z* = f(x*,y*) = " + zk + " .");
          }
        }
        else {
          if (normk <= epsilon)
            frame.algorithm_panel.output.append
              ("\n  " + "Abbruch des Algorithmus wegen ||grad(xk,yk)|| <= " + epsilon + ".");
          else
            frame.algorithm_panel.output.append
              ("\n  " + "Abbruch des Algorithmus wegen k >= " + kmax + ".");
               
          frame.algorithm_panel.output.append                                      
            ("\n  " + "Es wurde der Punkt (x*,y*) = ( " + xk + " , " + yk + " )" +
             "\n  " + "mit z* = f(x*,y*) = " + zk + " gefunden.");
              
          if (normk >= 0.01) {
            frame.algorithm_panel.output.append
              ("\n  " + "Der gefundene Punkt (x*,y*) ist kein kritischer Punkt der eingegebenen Funktion," +
               "\n  " + "da ||gradf(x*,y*)|| viel grsser als 0 ist.");
          }
          else {
	    if (normk <= epsilon)
              frame.algorithm_panel.output.append
                ("\n  " + "Der gefundene Punkt (x*,y*) ist ein kritischer Punkt der eingegebenen Funktion," +
                 "\n  " + "da ||gradf(x*,y*)|| <= " + epsilon + " gilt.");
            else
              frame.algorithm_panel.output.append
                ("\n  " + "Der gefundene Punkt (x*,y*) ist ein kritischer Punkt der eingegebenen Funktion," +
                 "\n  " + "da ||gradf(x*,y*)|| <= 1.0E-3 gilt.");
                  
            // Auf Definitheit testen	
            q11 = eval(parser_dxdx, xk, yk); if (error) return;
            q12 = eval(parser_dydx, xk, yk); if (error) return;
            q21 = eval(parser_dxdy, xk, yk); if (error) return;
            q22 = eval(parser_dydy, xk, yk); if (error) return;
            
            if ( (q11 > 0.0) && (q11*q22-q12*q21 > 0.0) )
              frame.algorithm_panel.output.append
                   ("\n  " + "Dieser kritische Punkt (x*,y*) ist ein strenges lokales Minimum der eingegebenen Funktion," +
                    "\n  " + "da die Hessematrix an dieser Stelle positiv definit ist.");
            else if ( (q11 >= 0.0) && (q22 >= 0.0) && (q11*q22-q12*q21 >= 0.0) )
              frame.algorithm_panel.output.append
                   ("\n  " + "Dieser kritische Punkt (x*,y*) kann ein strenges lokales Minimum der eingegebenen Funktion" +
                    "\n  " + "sein, da die Hessematrix an dieser Stelle positiv semidefinit ist.");
            else
              frame.algorithm_panel.output.append
                   ("\n  " + "Dieser kritische Punkt (x*,y*) ist kein strenges lokales Minimum der eingegebenen Funktion," +
                    "\n  " + "da die Hessematrix an dieser Stelle nicht positiv (semi)definit ist.");

          }

        }
      }
      stop();
      fertig = true;
      return;
    }
     
  }

  private void isConstant(String func) {
    for (int index=0; index<func.length(); index++) {
      if (func.charAt(index) == 'x') {
        if ( (index == 0) || ( (index > 0) && (func.charAt(index-1) != 'e') ) ) {
          error = true;
          break;
        }
      }
      else
      if (func.charAt(index) == 'y') {     	
          error = true;
          break;
      }
    }
  }
  
  private double eval(Parser parser, double x, double y) {
    error = false;
    parser.setVariable(1,x);
    parser.setVariable(2,y);
    
    double result = parser.evaluate();
    
    if (parser.getErrorCode() != parser.NO_ERROR) {
      if (trace) {
      	frame.algorithm_panel.output.append
      	  ("\n  " + "Speicher Fehler:" +
           "\n  " + "Bitte geben Sie eine andere Funktion ein und starten Sie den Algorithmus neu.");
        stop();
        k--;
      }
      fertig = true;
      error = true;
    }
    
    return result;
  }
  
  private double Newton() {
    int i;
    double t0;
    double g0t0, g1t0, hst0;
    double hsst0;
    double hst0m, hst0p;
    
    i = 0;
    t0 = 0.0;
    hst0 = hs0;
    hsst0 = 1.0;
    
    while ((i < 100) && (Math.abs(hst0) > 1.0e-05)) {
      
      i++;
      
      // Berechne (g0t0,g1t0) = grad(xk_ + t0*dk) = grad(xk + t0*ak, yk + t0*bk)
      g0t0 = eval(parser_dx, xk + t0*ak, yk + t0*bk); if (error) return 0.0;
      g1t0 = eval(parser_dy, xk + t0*ak, yk + t0*bk); if (error) return 0.0;
           
      // Berechne hst0 = h'(t0) = grad(xk_ + t0*dk)T * dk = (g0t0,g1t0)T * (ak,bk) 
      hst0 = g0t0*ak + g1t0*bk;
      
      // Berechne (q11,q12,q21,q22) = hesse(xk_ + t0*dk) = hesse(xk + t0*ak, yk + t0*bk)
      q11 = eval(parser_dxdx, xk + t0*ak, yk + t0*bk); if (error) return 0.0;
      q12 = eval(parser_dydx, xk + t0*ak, yk + t0*bk); if (error) return 0.0;
      q21 = eval(parser_dxdy, xk + t0*ak, yk + t0*bk); if (error) return 0.0;
      q22 = eval(parser_dydy, xk + t0*ak, yk + t0*bk); if (error) return 0.0;
      
      // Berechne hsst0 = h''(t0) = dk * hesse(xk_ + t0*dk) * (dk)T
      hsst0 = q11*ak*ak + q21*ak*bk + q12*ak*bk + q22*bk*bk;
      
      if (hsst0 == 0.0) return 0.0; 
      t0 = t0 - hst0/hsst0;
      if (t0 <= 0.0) return 0.0;
    }
//System.out.println(i + " " + t0);
 
    if (Math.abs(hst0) > 1.0e-03) return 0.0;  
 
    // t0 <= 0
    if (t0 <= 0.0) return 0.0;
    
    // hsst0 < 0 ==> t0 Maximum
    if (hsst0 < 0.0) return 0.0;
    
    // hsst0 = 0 ==> falls kein - + VZW, dann kein Minimum
    if (hsst0 == 0.0) {
    	
      // Berechne (g0t0,g1t0) = grad(xk_ + (t0-0.1)*dk)
      g0t0 = eval(parser_dx, xk + (t0-0.1)*ak, yk + (t0-0.1)*bk); if (error) return 0.0;
      g1t0 = eval(parser_dy, xk + (t0-0.1)*ak, yk + (t0-0.1)*bk); if (error) return 0.0;
           
      // Berechne hst0 = h'(t0-0.1) = grad(xk_ + (t0-0.1)*dk)T * dk = (g0t0,g1t0)T * (ak,bk) 
      hst0m = g0t0*ak + g1t0*bk;
      
      // Berechne (g0t0,g1t0) = grad(xk_ + (t0+0.1)*dk)
      g0t0 = eval(parser_dx, xk + (t0+0.1)*ak, yk + (t0+0.1)*bk); if (error) return 0.0;
      g1t0 = eval(parser_dy, xk + (t0+0.1)*ak, yk + (t0+0.1)*bk); if (error) return 0.0;
           
      // Berechne hst0 = h'(t0+0.1) = grad(xk_ + (t0+0.1)*dk)T * dk = (g0t0,g1t0)T * (ak,bk) 
      hst0p = g0t0*ak + g1t0*bk;
      
      if (!((hst0m < 0.0) && (hst0p > 0.0))) return 0.0; // t0 kein Minimum
    }
    
    return t0;
  }
  
  private double LineSearch() {
    // Nocedal-Wright Linien-Suche (S.59 ihres Buches "Numerical Optimization (1999)")
    // Bestimmt eine Schrittweite tk>0, welches die zwei strengen Wolfe-Powell-Bedingungen
    //     (1) h(tk) <= h(0) + p1*tk*h'(0)
    //     (2) |h'(tk)| <= p2*|h'(0)|
    // erfllt.
    // Falls tk=0 zurckgegeben wird, so konnte kein solches tk>0 gefunden werden kann.

    double t0, ht0, hst0;
    double t1, ht1, g0t1, g1t1, hst1;

    // Initialisiere t0=0, h(t0)=h(0), h'(t0)=h'(0) und t1=1
    t0 = 0.0; ht0 = h0; hst0 = hs0;
    t1 = 1.0;

    for (int i=1; i<=20; i++) {

      // Berechne ht1 = h(t1) = f(xk_ + t1*dk) = f(xk + t1*ak, yk + t1*bk)
      ht1  = eval(parser_f,  xk + t1*ak, yk + t1*bk); if (error) return 0.0;
      
      // Falls (h(t1) > h(0) + p1*t1*h'(0)) oder ((h(t1) >= h(t0) und (i > 1))
      //   return tk=Zoom(t0,t1)
      if ( (ht1 > h0 + p1*t1*hs0) || ((ht1 >= ht0) && (i > 1)) )
        return Zoom(t0,t1,ht0,ht1,hst0);
     
      // Berechne (g0t1,g1t1) = grad(xk_ + t1*dk) = grad(xk + t1*ak, yk + t1*bk)
      g0t1 = eval(parser_dx, xk + t1*ak, yk + t1*bk); if (error) return 0.0;
      g1t1 = eval(parser_dy, xk + t1*ak, yk + t1*bk); if (error) return 0.0;
           
      // Berechne hst1 = h'(t1) = grad(xk_ + t1*dk)T * dk = (g0t1,g1t1)T * (ak,bk) 
      hst1 = g0t1*ak + g1t1*bk;

      // Falls (h'(t1) >= 0)
      //   return tk=Zoom(t1,t0)
//System.out.println("W0: " + hst1 + " >= 0");
      if (hst1 >= 0)
        return Zoom(t1,t0,ht1,ht0,hst1);

      // Falls (|h'(t1)| <= -p2 * h'(0))
      //   return tk=t1
//System.out.println("W2: " + Math.abs(hst1) + " <= " + (-p2 * hs0));
      if (Math.abs(hst1) <= -p2 * hs0)
        return t1;

      // Nchste Iteration
      t0 = t1; ht0 = ht1;
      t1 = 2 * t1;
    }
    
    // tk=0, falls nach 10 Iterationen kein tk>0 gefunden werden konnte.
    return 0.0;
  }
  
  private double Zoom(double tlo, double thi, double htlo, double hthi, double hstlo) {
   
    double tj, htj, g0tj, g1tj, hstj;
    double c;
   
    for (int j=1; j<=70; j++) {

      // Interpolation:  c = (h(thi) - h(tlo) - (thi-tlo)*h'(tlo)) / (thi-tlo)^2
      c = (hthi - htlo - (thi-tlo)*hstlo) / ((thi-tlo)*(thi-tlo));
      
      if (c > 0.0)
        // Minimum der quadratischen Approximation: tj = tlo - h'(tlo)/(2*c)
        tj = tlo - hstlo/(2*c);
      else
        // Bisektion:  tj = (tlo+thi) / 2
        tj = (tlo+thi) / 2;

      // Berechne htj = h(tj) = f(xk_ + tj*dk) = f(xk + tj*ak, yk + tj*bk)
      htj  = eval(parser_f,  xk + tj*ak, yk + tj*bk); if (error) return 0.0;
      
      // Falls (h(tj) > h(0) + p1*tj*h'(0)) oder (h(tj) >= h(tlo))
      //   thi = tj und h(thi) = h(tj)
      if ( (htj > h0 + p1*tj*hs0) || (htj >= htlo) ) {
        thi = tj; hthi = htj;
      }
      else {
        // Berechne (g0tj,g1tj) = grad(xk_ + tj*dk) = grad(xk + tj*ak, yk + tj*bk)
        g0tj = eval(parser_dx, xk + tj*ak, yk + tj*bk); if (error) return 0.0;
        g1tj = eval(parser_dy, xk + tj*ak, yk + tj*bk); if (error) return 0.0;  
           
        // Berechne hstj = h'(tj) = grad(xk_ + tj*dk)T * dk = (g0tj,g1tj)T * (ak,bk) 
        hstj = g0tj*ak + g1tj*bk;
        
        // Falls (h'(tj) * (thi-tlo) >= 0)
        //   thi = tlo und h(thi) = h(tlo)
        if (hstj * (thi-tlo) >= 0) {
          thi = tlo; hthi = htlo;
        }        
        
        // Falls (|h'(tj)| <= -p2 * h'(0))
        //   return tk=tj
        if (Math.abs(hstj) <= -p2 * hs0)
          return tj;
      
        // tlo = tj und h(tlo) = h(tj) und h'(tlo) = h'(tj)
        tlo = tj; htlo = htj; hstlo = hstj;
      }
    }
    
   // tk=0, falls nach 50 Iterationen kein tk>0 gefunden werden konnte.
   return 0.0;
  }
  
}
